/*
  This file belongs in

  <TheGHOUL/DemoFiles/>

  Hank J. van Leuvensteijn Jr. 2021
  theghoul@hankjr.ca
*/

include<TheGHOUL/Config.scad>

// parameters ==================================================================

// size
BaseRadius=40;
Height=200;
RimRadius=30;
WallThickness=3;

// thickness of the horizontal 'slices' through the base
VerticalResolution=1;

// shape
Amplitude=20; // size (max-min) of the 'belly' and 'neck'
// angles, almost any value works, be bold ;-)
BStart=30; // initial  (base) 'angle' value for the sinusoid
BEnd=300; // final  (rim) 'angle' value for the sinusoid

// pattern
HelixAngle=360; // angle through which the helix twists from base to rim
HCount=15; // number of helices in the base
HGap=0.75; // 0.3 means 70% helix, 30% gap &c.

// =============================================================================

// angle 'included' by the helix width
HWAngle=360/HCount*(1-HGap);

// the outside radius of the lamp-base at 'height=Z'
// this essentially describes the sinusoid shape of the lamp-base
function OutRad(Z)=
    let(
        Frcn=Z/Height
    )
    BaseRadius  // start radius
    +(RimRadius-BaseRadius)*Frcn // 'slope' from start to end radius
    +(
        -sin(BStart)-(sin(BEnd)-sin(BStart))*Frcn // place sinusoid start and end on the 'slope' line
        +sin(BStart+(BEnd-BStart)*Frcn) // add sinusoid
    )*Amplitude // multiply sinusoid 'depth'
;

// create a cross-section through the helix at 'height=Z'
function Slice(Z)=
    let(
        ORad=OutRad(Z),
        IRad=ORad-WallThickness,
        Points=ceil(HWAngle/360*Segments(ORad)), // number of points on the outside/inside arc of the polygon
        Facet=HWAngle/Points // facet size (angle)
    )
    [
        for(Idex=[0:1:Points])
            [ORad*cos(Idex*Facet),ORad*sin(Idex*Facet),Z], // outside points
        for(Jdex=[Points:-1:0])
            [IRad*cos(Jdex*Facet),IRad*sin(Jdex*Facet),Z], // inside points
    ]
;

function LeftHelix(HelixAngle=HelixAngle)=
    [
        for(Idex=[0:VerticalResolution:Height])
            AffineRotate([0,0,Idex/Height*HelixAngle],Slice(Idex)) // stack polygons into a ccw helix
    ]
;

function RightHelix(HelixAngle=-HelixAngle)=
    [
        for(Idex=[0:VerticalResolution:Height])
            AffineRotate([0,0,Idex/Height*HelixAngle],Slice(Idex)) // stack polygons into a cw helix
    ]
;

// bottom
cylinder(h=WallThickness,r1=BaseRadius,r2=OutRad(WallThickness));

// lid
translate([0,0,Height-WallThickness])
cylinder(h=WallThickness,r1=OutRad(Height-WallThickness),r2=OutRad(Height));

CircularArray(Radius=0,IncludedAngle=360,Number=HCount)
CoverMesh(LeftHelix());

CircularArray(Radius=0,IncludedAngle=360,Number=HCount)
CoverMesh(RightHelix());
