/*
  The basic animation function. Like $t is the fraction of the entire ANIMATION
  SEQUENCE in OpenSCAD, 'AnimationFxFraction()' returns the fraction of the
  OBJECT PERIOD, which is the period over which the object exists and is
  animated.

  Initial value is -1, switches to 0 at Create, increases linearly to 1 from Start to Finish, stays at 1 until Delete when it drops to -2.
  *Frequency*> 1 creates a saw-tooth-like triangular (wave) signal between *Start* and *Finish*.

  AnimVector    :   [Create,Start,Finish,Delete]
  Frequency     :   Creates a recurring saw-tooth Factor (slope-up to 1 with a
                    sharp drop-off to 0) with *Frequency* peaks or cycles during the object animation, to get a 'Hertz' (CPS) like frequency of X repetitions per $AnimSeg, simply multiply *Frequency* with the duration of the animation, i.e., (Finish-Start).
*/
function AnimationFxFraction(AnimVector,Frequency=1)=
    let(
        /* Populate some variables. */
        Now=$t*$AnimSegs,
        Create=AnimVector[0],
        Start=AnimVector[1],
        Finish=AnimVector[2],
        Delete=AnimVector[3]==undef?$AnimSegs:AnimVector[3],
        /* Fraction of object animation at Frequency. */
        Fraction=((Now-Start)/(Finish-Start)*Frequency)%1
    )
    /* Catch negative frequencies. */
    Frequency<0?undef:
    /* Do we exist? */
    Now<Create?-1:
    /* Now we do! */
    Now<Start?0:
    /* Are we changing? */
    Now<Finish?min(1,max(0,Fraction)):
    Now<Delete?1:
    /* Are we deleted? Giving post-deletion a different value from pre-creation allows for differentiation elsewhere, if desired. */
    -2
;

// Shorthand version:
function AnFxFr(AnimVector,Frequency=1)=
    AnimationFxFraction(AnimVector,Frequency)
;

