/*
  This is the main animation signal generator. It is highly customisable and has it's own Demo File; it pays off to check that out...

  The sum of two sinusoid functions, best results are achieved with a 'Domain' of 90 degrees for the first and 180 for the second, creating a kind of 'hump on a hump' effect.

  The function has a *Base*-line; this lifts the whole function up by the value of *Base*. You may prefer to reduce the amplitude of the remainder to keep the range [0,1].

  Using a 180 degree *Domain* for the second function adds a 'bump' to the curve, this looks good for sphere sizes, overshooting and then returning to their size proper.

  Each function has it's own *SinVector* (more below) defining the curve.

  The *Amplitude* speaks for itself.

  *Domain* is expressed in degrees, a *Domain* of 180 means a 180 degree sinusoid will be generated within the length of the animation segment.

  Odd *Power*s can be used the same as even *Power*s, because (usually) only 90 or 180 degrees of sinusoid are used, and the negative part of the curve (which--of course--happens above 180 degrees) never comes into play.
  Higher *Power* gives narrower bumps for the 180 degree domain curve, and bigger delays before the 90 degree domain curve starts to rise.

  'AnimationFxTwoSinesDemo.scad' in the DemoFiles directory, illustrates the use of this function. Go play around with it or go to

  https://www.desmos.com/calculator/rcp4vap7zl

  or a similar site to see the effects of the different parameters.
  If the above link doesn't work anymore, here's the function to enter in desmos' calculator; you'll figure out which parameter is what, I'm sure:

  1\cdot\sin\left(x\cdot90\right)^{6}+1\cdot\sin\left(x\cdot180\right)^{6}

  The parameter structure:
    AnimVector      : [Start,Finish,Delete]
    SinVectorA/B    : [Amplitude,Domain,Power]
    Max             : Value to clip the top of the curve.

  Some examples:
    Base    SinVectorA  SinvectorB
    0       [1,90,10]   [1.5,180,6]     :   Bounce-size for spheres.
    0       [1,90,6]    [0.8,180,6]     :   Bump-size for cylinders.
    0.3     [0.7,90,20] [0,180,2]       :   Alpha, .
    0       [0,90,10]   [3,180,6]       :   With Max=1, for Fade-in Fade-out.

*/
function AnimationFxTwoSines(AnimVector,Base=0,
                           SinVectorA=[1,90,1],
                           SinVectorB=[0,180,1],
                           Frequency=1,Max=10)=
    let(
        /* Populate some variables. */
        Now=$t*$AnimSegs,
        Create=AnimVector[0],
        Start=AnimVector[1],
        Finish=AnimVector[2],
        Delete=AnimVector[3]==undef?$AnimSegs:AnimVector[3],
        /* Fraction of object animation at Frequency. */
        Frac=min(1,(Now-Start)/(Finish-Start)),
        AmpA=SinVectorA.x,
        DomA=SinVectorA.y,
        PowA=SinVectorA.z,
        AmpB=SinVectorB.x,
        DomB=SinVectorB.y,
        PowB=SinVectorB.z
    )
    /* Do we exist? */
    Now<Create?-1:
    /* Now we do! */
    Now<Start?Base:
    /* Are we changing? */
    Now<Delete?min(Base+AmpA*pow(sin(Frac*DomA),PowA)
                +AmpB*pow(sin(Frac*DomB),PowB),Max):
    /* Are we deleted? Giving post-deletion a different value from pre-creation allows for differentiation elsewhere, if desired. */
    -2
;

/* Shorthand version. */
function AnFxTS(AnimVector,Base=0,SinVectorA=[1,90,1],SinVectorB=[0,180,1],
                Frequency=1,Max=10)=
    AnimationFxTwoSines(AnimVector,Base,SinVectorA,SinVectorB,Frequency,Max)
;

// COSINES =====================================================================

/* Not as useful, or easy to use, as AnFxTS(), mostly because the cosine has it's maximum at X=0, unlike the sine. It's here anyway, probably because I'd feel my work would be incomplete if it wasn't; at any rate, I'm sleeping fine, and that's what matters... */
function AnimationFxTwoCos(AnimVector,Base=0,
                           CosVectorA=[1,90,1],
                           CosVectorB=[0,180,1],
                           Frequency=1,Max=10)=
    let(
        Now=$t*$AnimSegs,
        Create=AnimVector[0],
        Start=AnimVector[1],
        Finish=AnimVector[2],
        Delete=AnimVector[3]==undef?$AnimSegs:AnimVector[3],
        /* Fraction of object animation at Frequency. */
        Frac=min(1,(Now-Start)/(Finish-Start)),
        AmpA=CosVectorA.x,
        DomA=CosVectorA.y,
        PowA=CosVectorA.z,
        AmpB=CosVectorB.x,
        DomB=CosVectorB.y,
        PowB=CosVectorB.z
    )
    /* Do we exist? */
    Now<Create?-1:
    /* Now we do! */
    Now<Start?Base:
    /* Are we changing? */
    Now<Delete?min(Base+AmpA*pow(cos(Frac*DomA),PowA)
                +AmpB*pow(cos(Frac*DomB),PowB),Max):
    /* Are we deleted? Giving post-deletion a different value from pre-creation allows for differentiation elsewhere, if desired. */
    -2
;

/* Shorthand version. */
function AnFxTC(AnimVector,Base=0,CosVectorA=[1,90,1],CosVectorB=[0,180,1],
                Frequency=1,Max=10)=
    AnimationFxTwoCos(AnimVector,Base,CosVectorA,CosVectorB,Frequency,Max)
;
