/*
  Called 'StringTee' after $t...

  Quick and dirty animation helper.

  StringTee(Start,Finish,BeginValue,EndValue)

  Start, Finish, time values [0:$AnimSegs], see $AnimSegs.txt in the animation library directory.
  BeginValue, EndValue, scalars or vectors, tuples etc.

  The returned value starts at 'BeginValue' when $t=0, grows linearly from 'BeginValue' to 'EndValue' between 'Start' and 'Finish', and remains at 'EndValue' until $t=1.

   $t*$AnimSegs=    0      Start   Finish  $AnimSegs
                    |          |   |          |
        EndValue                   ------------
                                  /
                                 /
                                /
        BeginValue  ------------

  Let's assume an $AnimSegs value of 10 (seconds).

  StringTee() is meant for quick-and-dirty-simple-animations, but if you'd like to get a bit more complex:

  To change the alpha value of a color from 0.2 at 5 seconds to 0.6 at 7 seconds and back to 0.4 by 8 seconds:

  //               S1  F1  Bgn1 End1           S2  F2  Bgn2 End2
  Alpha1=StringTee( 5,  7, 0.2, 0.6)+StringTee( 7,  8,  0,  -0.2);

  Time      Alpha1
  ------------------------------------------------------
  0         0.2     Bgn1 + Bgn2
  ...
  5         0.2     Bgn1 + Bgn2
  6         0.4     (Bgn1 linear to End1) + Bgn2
  7         0.6     End1 + Bgn2
  7.5       0.5     Max 1 + (Bgn2 linear to End2)
  8         0.4     End1 + End2
  ...
  10        0.4     End1 + End2
*/

/*
  The StringTee() function returns, as they do, a value.
*/
function StringTee(Start,Finish,BeginValue,EndValue)=
    let(
        Now=$t*$AnimSegs
    )
    Start>Now
    ?   BeginValue
    :   Finish>Now
        ?   (Now-Start)/(Finish-Start)*(EndValue-BeginValue)+BeginValue
        :   EndValue
;

/*
  The StringTee() module shows children when *Now* lies after *Start* and before *Finish*, and before or if *Preexisting* is _true_, as well as after when *Enduring* is _true_. It also makes $StValue available to the children(), useful methinks; almost as cool as Animate().
*/
module StringTee(Start,Finish,BeginValue,EndValue,Preexisting=false,Enduring=true){
    /* Check the time. */
    Now=$t*$AnimSegs;
    /* Store the value. */
    $StValue=StringTee(Start,Finish,BeginValue,EndValue);
    /* Show the kids. */
    if( ((Now<Start)&&Preexisting)||
        (Start<=Now&&Now<=Finish)||
        ((Now>Finish)&&Enduring)
    )
        children();
}


