/*

A quick 'howto':

Animation is'nt too hard in OpenSCAD, and even easier with the Animation library from The GHOUL. Rotation and translation are simple to do, but what about zooming in and out? Now you need to mess with $vpt, $vpr and $vpd... ViewPorts makes that easy. In smooth linear interpolations it takes you from one ViewPort to the next, have a look.

Also check out ViewPortsAnimationDemo.scad in the GHOUL/DemoFiles folder.

ViewPort must be declared like:

    ViewPort=
    ViewPorts([
        [0,[0,0,0],[60,0,45],8],
        [1.0,[0,15,7],[30,0,90],50],
    ...
        [9.0,[-5,15,8],[80,0,60],55],
        [10.0,[-8,27,9],[120,0,110],12]
    ]);

// Then do this:

    $vpt=ViewPort[1];
    $vpr=ViewPort[2];
    $vpd=ViewPort[3];

// You need something to look at:

    cube([20,20,20]);
    color("red")
    sphere(r=10);

// Now select 'View' -> 'Animate' and at the bottom of the window
// set 'Steps' to 100 and 'FPS' to 10, and watch the magic.
*/

/*
  ViewPorts, array of ViewPorts, each ViewPort is defined as:

  [Time,[$vpt],[$vpr],$vpd]

  For info on $vpt, $vpr and $vpd, see the OpenSCAD manual.

  The 'Time' parameter is simply a way to break up the animation in steps, depending on the 'FPS' and 'Step' settings in the Animation View, also see $AnimSegs.txt in this directory (TheGHOUL/Lib/Animation/).
  It MUST start at 0, and it ends at $AnimSegs (or before, see in-function documentation below), and it MUST be ordered, i.e. the next must be higher than the previous in the list, time is like that, you see...

  To stand still at a ViewPort, simply repeat it in the list, only increasing the Time parameter, say you want to stop at the second ViewPort, the list will look like this: [[ViewPort1],[ViewPort2],[ViewPort2],[ViewPort3],...] and the times can be something like 0,1,4,5,6... Now you will look at ViewPort 2 from Time=1 till Time=4.

  ViewPort() performs a linear interpolation between ViewPorts. If two ViewPorts are 180 degrees offset, consider the direction of rotation. Going from [...,...,180] to [...,...,0] is opposite to going from [...,...,180] to [...,...,360]. OpenSCAD doesn't care how big you make the angles, want to keep turning the same way? Just keep increasing the angle.

  Have fun.
*/
function ViewPort(ViewPorts)=
    let(
        /* What's our current 'time'? (Total duration is 0...$AnimSegs.) */
        TNow=$t*$AnimSegs,
        /* Between which two ViewPorts are we currently? */
        Previous=FindLTE(TNow,ViewPorts,Element=0,Type="Last"),
        /* When *ViewPorts* has not been defined for the total duration of $AnimSegs, make it cyclic. */
        Next=Dex(Previous+1,Len(ViewPorts)),
        TNxt=Next==0?$AnimSegs:ViewPorts[Next][0],
        TPrv=ViewPorts[Previous][0]
    )
    Intermediate(
        ViewPorts[Previous],ViewPorts[Next],
        /* Fraction of current interval: */
        (TNow-TPrv)/
        /* Total length of current interval: */
        (TNxt-TPrv)
    )
;

/*
  Helpers, pull or echo the ViewPort details, can be used when the status bar is too short to display everything, also useful when creating a sequence: Create a view, copy the ViewPort details and paste into your sequence...
  * The 'ViewPort()' format is maintained, the 'Time' parameter is simply populated with 0.
*/
module EchoViewPort(){
    Print(["ViewPort= ",[$t,$vpt,$vpr,$vpd]]);
}
/* Not sure if this is useful or not :-\ */
function GetViewPort()=
    [$t,$vpt,$vpr,$vpd]
;
