/*

IMPORTANT: =====================================================================
  The routines in this file are all NON-USER routines; you will never address them directly.
================================================================================

  Bézier curves, named after Mr. Pierre Bézier, who used them to design automobile bodies at Renault, are based on de Casteljau’s algorithms, developed by Paul de Casteljau in 1959, who became the first to apply them to computer-aided design at French automaker Citroën. Trust the French to be the first to develop a smart way to mathematically describe flowing auto-body lines.

  In The GHOUL, the following naming conventions are maintained:

  Tangent Point         A known (given) point _through which_ the curve passes,
                        and _to which_ therefore, there exists a (local) tangent (plane).

  Tangent point definitions look like:
  [[Vertex],[FM,INC,AZ],[RM,INC,AZ],[NM,INC,AZ],[PM,INC,AZ]]
  where:
  * [Vertex]            The Tangent Point itself, a 3D vertex.
  * INC                 Inclination of vector.
  * AZ                  Azimuth of vector.
  * FM                  Vector _length_ pointing Forwards to the next Vertex in
                        the curve or 3D cross-section.
  * RM                  Vector _length_ pointing in Reverse to the previous
                        Vertex in the curve or 3D cross-section.
  * NM                  Vector _length_ pointing towards the Next 3D
                        cross-section.
  * PM                  Vector _length_ pointing towards the Previous 3D
                        cross-section.

  Tangent Point Set     List of Tangent Points (see above), describing a
                        Curve, Shape or 3D Solid cross-section. In addition to the Tangent Points, this contains a (set of) Transformations that move and rotate the TPS within 2D or 3D, as required.

  Tangent Point Array   Array of Tangent Point Sets, describing a compound
                        Curve, multi-path Shape, or a 3D solid.

  TV, TVS, TVA          Tangent Vector equivalents of the above.
                        Tangent Vectors are Tangent Points, fully developed with transformations applied. They form the basis for the Bézier curve control point polygons that ultimately are the basis for the final curves. They are an intermediate step, and never directly specified by the user.

  Control Point         A Bézier curve is described by control points, two
                        points yield a straight line, any more generate a curve. The most convenient curve has 4 CP; the two endpoints, and two intermediate points, which determine the tangent with which the curve leaves the closest endpoint. This is a 'cubic' Bézier curve, and it is highly suited to describe smooth curves and solids.

  Control Polygon       List of Control Points, usually 4 (for a cubic Bézier
                        curve) describing a single Bézier curve.

  Control Polygon Set   List of Control Polygons, describing a
                        Curve, Shape or 3D Solid cross-section.

  Control Polygon Array Array of Control Polygon Sets, describing a compound
                        Curve, multi-path Shape, or a 3D solid.

NOTE: ==========================================================================
  Generally speaking, a curve, shape or 3D solid starts with tangent Point definitions, They are translated into Tangent Vector records, which are translated into Control Polygons, which form the basis for the Bézier curves that make up the diverse forms of curves, compound Curves, Shapes, multi-path Shapes and 3D Solids.
================================================================================

*/

/**
    Structures.

    {}:         Indicates optional parameters (3D).
    vertex:     [x,y,z]
    spherical:  [magnitude,inclination,azimuth] The GHOUL uses the 'physics
                convention', as opposed to the 'maths convention' which has the order of the angles reversed.

    // Tangent Point.
    [
        [vertex],[spherical],[spherical]{,[spherical],[spherical]}
    ]

    // A Set makes a complex Curve, Shape, or 3D cross-section. Includes optional Transforms.
    [ // Tangent Point Set.
        [ // Tangent Points.
            [
                [vertex],[spherical],[spherical]{,[spherical],[spherical]}
            ]
            , ...
        ],
        [ // Transforms.
            [...,"T"],[...,"R"]
        ]
    ]

    // An Array makes a Solid, a compound Curve, or a multi-path Shape.
    *  Solid:           Each set forms a cross section, or plane through
                        the solid.
    *  Compound curve:  Each set forms a partial curve. The curves are
                        daisy-chained.
    *  Shape:           Each set forms a closed curve. The curves are
                        subtractive where they overlap.
    [ // Tangent Point Array. (Curve, Shape or Solid)
        [ // Tangent Point Set. (Curve or Plane or 3D cross-section)
            [ // Tangent Points.
                [
                    [vertex],[spherical],[spherical]{,[spherical],[spherical]}
                ]
                , ...
            ],
            [ // Transforms.
                [...,"T"],[...,"R"]
            ]
        ]
        , ...
    ]

    // CPx follow the same pattern.
    [ // Control Polygon Array. (Curve, Shape or Solid)
        [ // Control Polygon Set. (Curve, Shape or Solid cross section)
            [ // Always present, 'Current' Curve or Plane.
                [
                    [vertex],[vertex],[vertex],[vertex]
                ]
                , ...
            ]
            {,[ // Only for 3D, Next Plane.
                [
                    [vertex],[vertex],[vertex],[vertex]
                ]
                , ...
            ]}
            {,[ // Only for 3D, Previous Plane.
                [
                    [vertex],[vertex],[vertex],[vertex]
                ]
                , ...
            ]}
        ]
    ]

**/

/* Outside of BezierShape(), $BezierShape=false, this way, we can let routines know that they were called by BezierShape() by setting $BezierShape=true inside BezierShape() and testing for it in the subsequent routines. This is required, so that routines called by BezierShape() know to close all curves, regardless of the presence of the 'open curve' indicator in any TPS. */
$BezierShape=false;

/*
  FROM TANGENT POINTS TO TANGENT VECTORS =======================================
*/

/*
  From TPA to TVA.

  Intermediate, non-user routine.
*/
function BezierTangentVectorArray(TPA)=
    [
        for(TPS=TPA)
        BezierTangentVectorSet(TPS)
    ]
;

/*
  From TPS to TVS. Since TPS contains transforms, they are applied here, so they are expressed in the resulting TVS--and the following CPS and curves &c.

  Intermediate, non-user routine.
*/
function BezierTangentVectorSet(TPS)=
    let(
        /* Check if curve is (explicitly) open. If the last Tangent Point in a Set is followed by [], this signifies a curve that is NOT closed, i.e., ends at the last TP. */
        Open=Last(TPS[0])==[],
        LST=Len(TPS[0])-(Open?1:0),
        Transforms=TPS[1],
        /* Vectors get only rotations, no translation. */
        Rotations=[for(Transform=Transforms)if(Transform[3]=="R")Transform],
        RawVectors=concat(
            [for(Idex=[0:LST])BezierTangentVector(TPS[0][Idex])],
            /* Close the curve if not explicitly open, if this is a BezierShape() call, explicit open is ignored. */
            (Open&&!$BezierShape)?[]:[BezierTangentVector(TPS[0][0])]
        )
    )
    [ /* For each TP. */
        for(Record=RawVectors)
            [  /* Apply transforms. */
                /* The vertex gets translations AND rotations. */
                AffineTransform(
                    TransformMatrix(Transforms)
                    ,Record[0]
                ),
                /* Vectors don't get translated, they're NOT vertices, but they DO need to be rotated. */
                for(Jdex=[1:Len(Record)])
                    AffineTransform(
                        TransformMatrix(Rotations)
                        ,Record[Jdex]
                    )
            ]
    ]
;

/*
  Creates a Tangent Vector Record for a Tangent Point. This is the Vertex and a set of curvature Vectors. The Vertex is the actual Tangent Point, i.e., the vertex of the curve for which the tangent is given. The curvature Vectors describe the tangent to the curve at the Vertex, in Forward and Reverse direction, and if the TP is in 3D, also in the direction of the Next and Previous TP Set. The magnitude of the tangent vectors describes the 'tendency' of the curve (or surface) to continue along the tangent, i.e., the greater the tangent vector's magnitude, the wider (larger radius) the initial curvature, a magnitude of _0_ results in a sharp corner.

  Accepts 2D and 3D Tangent Points, and generates the appropriate Vector record.

  Intermediate, non-user routine.
*/
function BezierTangentVector(TP) =
    let(
        /* Forward spherical, always present. */
        FM=TP[1][0],
        FINC=TP[1][1],
        FAZ=TP[1][2],
        /* Reverse spherical, if missing copy Forward data. */
        RM=undef==TP[2][0]
        ?   FM
        :   TP[2][0],
        RINC=undef==TP[2][1]?FINC-180:TP[2][1],
        RAZ=undef==TP[2][2]?FAZ:TP[2][2],
        /* Everything from here on is _undef_ in 2D. */
        /* Next (plane) spherical, must be present for 3D, if absent we're not making a solid, just leave it _undef_. */
        NM=TP[3][0],
        NINC=undef==NM
        ?   undef
        :   undef==TP[3][1]
            ?   0 // NM is spec'd but not NINC.
            :   TP[3][1],
        NAZ=undef==NM
        ?   undef
        :   undef==TP[3][2]
            ?   0 // NM is spec'd but not NAZ.
            :   TP[3][2],
        /* Previous (plane) spherical, if missing copy Next data. */
        PM=undef==TP[4][0]
        ?   NM
        :   TP[4][0],
        PINC=undef==TP[4][1]
        ?   undef==NINC // Can't do [undef-180], so, error trap.
            ?   undef
            :   NINC-180
        :   TP[4][1],
        PAZ=undef==TP[4][2]?NAZ:TP[4][2],
        /* The tangent point vectors. */
        FV=AffineRotate([0,FINC,FAZ],[0,0,FM]),
        RV=AffineRotate([0,RINC,RAZ],[0,0,RM]),
        NV=undef==NM
        ?   undef
        :   AffineRotate([0,NINC,NAZ],[0,0,NM]),
        PV=undef==PM
        ?   undef
        :   AffineRotate([0,PINC,PAZ],[0,0,PM])
    )
    [
        TP[0],  // Vertex, i.e., base vector.
        FV,     // Forward magnitude vector.
        RV,     // Reverse magnitude vector.
         if(undef!=NM)
            NV, // Next plane magnitude vector.
         if(undef!=PM)
            PV, // Previous plane magnitude vector.
    ]
;

/*
  FROM TANGENT VECTORS TO CONTROL POLYGONS =====================================
*/

/*
  From TVA to CPA.

  Intermediate, non-user routine.
*/
function BezierControlPolygonArray(TVA)=
    [
        for(TVS=TVA) // For each curve.
        BezierControlPolygonSet(TVS)
    ]
;

/*
  From TVS to CPS.

  Each curve may contain several partial curves. Because solids also have a 'next plane' and 'previous plane' curve, these curves are wrapped in another set, like:

  [
    // Current plane.
    [[partial curve],[partial curve],[partial curve],...],**
    // Next plane.
    [[partial curve],[partial curve],[partial curve],...],
    // Previous plane.
    [[partial curve],[partial curve],[partial curve],...]
  ]

  To keep structures _consistent_, this is also the case for regular curves (2D and 3D) as well as shapes (always 2D), which therefore have a 'redundant' set of brackets and look like:

  [
    // The one curve.
    [[partial curve],[partial curve],[partial curve],...],
  ]

  It means we can use the same routines to treat a CPS, whether it describes a simple curve, a complex curve, a shape or a solid. Tidy.

  ** Note: 'partial curve' in the above examples should be read as 'partial curve control point polygon'.

  Intermediate, non-user routine.
*/
function BezierControlPolygonSet(TVS)=
    let(
        // 3D solid flag.
        Solid=len(TVS[0])>3,
        // Last partial curve start vectors--end vectors are at len(TVS)-1.
        Imax=len(TVS)-2
    )
    // We're looking at a single TP? C'mon user!
    Imax<0?undef:
    [
        // Curve or Current Plane.
        [
            for (Idex=[0:Imax]) // A Control Polygon for each Vector pair.
                BezierControlPolygon(TVS[Idex],TVS[Idex+1])
        ],
        if(Solid)
        // Next Plane.
        [
            for (Idex=[0:Imax]) // See above.
                BezierControlPolygon(TVS[Idex],TVS[Idex+1])
                + [TVS[Idex][3],TVS[Idex][3],TVS[Idex+1][3],TVS[Idex+1][3]]
        ],
        if(Solid)
        // Previous Plane.
        [
            for (Idex=[0:Imax]) // See above.
                BezierControlPolygon(TVS[Idex],TVS[Idex+1])
                + [TVS[Idex][4],TVS[Idex][4],TVS[Idex+1][4],TVS[Idex+1][4]]
        ]
    ]
;

/*
  Takes two Tangent Vector records and returns the control points for the curve section between them.

  Note: Because DeCasteljau() is just an interpolation, this routine can treat cross-section curves--when generating the 'vertical' curves through 3D solids--just like it treats vertices when generating regular curves. Magic.

  Intermediate, non-user routine.
*/
function BezierControlPolygon(TV1,TV2)=
    [
        TV1[0]       , // Start vector vertex.
        TV1[0]+TV1[1], // Start vector vertex plus forward vector.
        TV2[0]+TV2[2], // End vector vertex plus reverse vector.
        TV2[0]         // End vector vertex.
    ]
;

/*
  HELPER FUNCTION ==============================================================
  Not used by by the routines here, but by BezierCurve() &c.

  Does what it says on the tin; returns Boolean indicating whether *Array* is a TPA or not.

  A TPA is most easily identified by the fact that it either has an empty *Transforms* list, or that it has a list with a string as element 4 in the location of a Transform.
*/
function IsTPA(Array)=
    Array[0][1]==[] // Empty *Transforms* list.
    ||
    is_string(Array[0][1][0][3]) // String element in Transform list element.
;
// Some tests.
//echo(IsTPA([[[],[[1,2,3,"T"]]]]));
//echo(IsTPA([[[],[]]]));
//echo(IsTPA([[[],[[1,2,3]]]]));
