/*
  Create a 3D Bézier curve from a (set of) control-point-set(s) or a TPA.

    function TPS(
            Transforms=[[0,0,0,"T"]]

        )=
            [           // TPSet
                [       //  TPs
                    [[vertex],[FM,INC,AZ],[RM,INC,AZ]]
                   ,[[vertex],[FM,INC,AZ],[RM,INC,AZ]]
                   ,...
                   ,[] // Open curve indicator--optional.
                ],
                Transforms
            ]

    ;

    TPA=[               // TPArray
            TPS()
           ,TPS()
           ,...
        ]
    ;

  TPA        : An array of tangent point sets.
  Resolution : The number of line segments that will be generated on each curve.
  Closed     : Boolean, close the curve between the last & first TP if _true_.
  Single     : Boolean, return a single curve (not a set of partial curves)
               if _true_.

  BezierCurve() will also accept a single CPP (Control point polygon), or a list of CPPs, even a list of lists of CPP, instead of a TPA, however, the list of lists has one peculiarity; each list in the list needs double brackets, this is because BezierControlPolygonArray() generates three 'curves' for each TPS in a TPA when a 3D solid is processed, and to keep things the same across the routines, this is the case even when we're looking at single curves; consistency to the point of obscurity I'm afraid... Luckily, BezierCurve() does the thinking for you and treats whatever you feed it with consideration, care and tenderness.
*/
function BezierCurve(TPA,Resolution=20,Single=true)=
    let(
        CPA=IsTPA(TPA)
        // We have a TPA. Hopefully. Otherwise, utter chaos.
        ?   BezierControlPolygonArray(BezierTangentVectorArray(TPA))
        :   is_num(TPA[0][0])?[[[TPA]]] // Control polygon.
            :   is_num(TPA[0][0][0])?[[TPA]] // Curve (multiple polygons)
                :   is_num(TPA[0][0][0][0])?[for(CPS=TPA)[CPS]] // Multiple curves.
                    :   undef,
        Curves=[
            // Curve set.
            for(Idex=[0:Len(CPA)])
                // A curve.
                RemoveDoubles(
                    [ // For each partial curve.
                        for(Jdex=[0:Len(CPA[Idex][0])])
                            // For each resolution point.
                            for(Vertex=[0:Resolution])
                                DeCasteljau(CPA[Idex][0][Jdex]
                                    ,Vertex/(Resolution))
                    ]
                )
        ]
    )
    Single
    ?   RemoveDoubles(FlattenArray(Curves))
    :   Curves
;

/*
  The module shows the curve; should this be in the /Visualisers/ directory? Oh well... It's basically ShowCurve() with Bézier input instead of a vertex array, see there for parameter documentation. If you need to...
*/
module BezierCurve(TPA,Resolution=30,Closed=false,Single=true,Radius=0.1,Frac=1,Connect=true){
    Curve=BezierCurve(TPA,Resolution,Single);
    if(is_num(Curve[0][0])) // Do we have a single curve?
        ShowCurve(Curve,Radius,Frac,Connect,Closed);
    else // We must have multiple curves.
    for (Part=Curve)
        ShowCurve(Part,Radius,Frac,Connect,Closed);
}
