/*
  Create a 2D Bézier curve from a (set of) tangent-point-set(s).
  TPA  :       A  shape tangent point set like this:

TPA=[
        // TPS
        [
            [ // TPs
                [[Vertex],[FV],{[RV]}],
                [[Vertex],[FV],{[RV]}],
                ...,
                [] // Empty TP signifies open curve, BezierShape() ignores this!
            ],
            [] // Shapes ignore Transforms, however, an empty transform
        ],     // is required for compatibility.
        ...
    ]
;

  TPA        : An array of curve tangent point sets.
  Resolution : The number of line segments that will be generated on each curve.

  Multiple TPSs will create multiple (additive/subtractive) paths for the Shape polygon.

  BezierShape() will also accept a single CPP (Control point polygon), or a list of CPPs, even a list of lists of CPP, instead of a TPA, however, the list of lists has one peculiarity; each list in the list needs double brackets, this is because BezierControlPolygonArray() generates three 'curves' for each TPS in a TPA when a 3D solid is processed, and to keep things the same across the routines, this is the case even when we're looking at single curves; consistency to the point of obscurity I'm afraid... Luckily, BezierShape() does the thinking for you and treats whatever you feed it with consideration, care and tenderness.
*/
function BezierShape(TPA,Resolution=20)=
    let(
        /*
          Because we are (re)using the Curve routines, the $BezierShape parameter is required for BezierTangentVectorArray(). It causes TP[n] and TP[0] to be seen as the last pair, which results in a _closed_ path, which, naturally, is required for a polygon. This happens even when the TPS ends with the 'open curve' indicator [].
        */
        CPA=IsTPA(TPA)
        // We have a TPA. Hopefully. Otherwise, utter chaos.
        ?   BezierControlPolygonArray(BezierTangentVectorArray(TPA),$BezierShape=true)
        :   is_num(TPA[0][0])?[[[TPA]]] // Control polygon.
            :   is_num(TPA[0][0][0])?[[TPA]] // Curve (multiple polygons)
                :   is_num(TPA[0][0][0][0])?[for(CPS=TPA)[CPS]] // Multiple curves.
                    :   undef,
        Paths=
            [ // Path Array, primary path, plus subtractive secondaries.
                for(Idex=[0:Len(CPA)])
                    CropArray(RemoveDoubles(
                        [ // Each CPS, whether compound or not, becomes a path.
                            for(Jdex=[0:Len(CPA[Idex][0])])
                                for(Vertex=[0:Resolution])
                                    DeCasteljau(CPA[Idex][0][Jdex]
                                        ,Vertex/(Resolution))
                        ]
                    ))
            ],
        NPaths=Len(Paths)
    )
    [ // TheGHOUL polygon [[vertices],[paths]].
        // Vertices.
        FlattenArray(Paths),
        // Paths.
        concat( // Main path.
                [
                    RangeToTuple([0:Len(Paths[0])])
                ],
                // Additional paths, if present.
                NPaths==0?[]:
                [
                    for(Idex=[1:NPaths])
                        ArrayAdd(
                            RangeToTuple([0:Len(Paths[Idex])])
                            ,ArraySum(
                                [
                                    for(Jdex=[0:Idex-1])
                                        len(Paths[Jdex])
                                ]
                            )
                        )
                ]
        )
    ]
;

/*
  Like the function, but display.
*/
module BezierShape(TPA,Resolution=20){
    Shape=BezierShape(TPA,Resolution);
    polygon(Shape[0],Shape[1],10);
}
/*
  Yes... Just an alibi, perhaps for backward compatibility...
*/
module BezierPolygon(TPA,Resolution=20){
    Shape=BezierShape(TPA,Resolution);
    polygon(Shape[0],Shape[1],10);
}

