/*
  Generate and display the bézier curve based solid (polyhedron).

  TPA           : Tangent Point Array. An array of Tangent Point Sets, each
                  forming a cross-section through the solid.
  Resolution    : Number of points per generated Bézier curve.
  * Section     : Horizontal 'Cross section' through the solid.
  * Set         : Vertical curves between sections.

  BezierSolid() accepts only a TPA. Bulding a CPA for a solid is probably fool's work anyway; I doubt I'll ever make this 'CPA friendly', but I've proven myself wrong before...
*/
module BezierSolid(TPA,SectionResolution=30,SetResolution=20,EndCaps=true,Doughnut=false){
    CoverMesh(
        BezierSolid(TPA,SectionResolution,SetResolution,Doughnut),
        EndCaps,Doughnut);
}

/*
  Each TPS yields a triplet of 'cross-section curves'
  [[Current], [Next], [Previous]]
  These are rearranged into 'quads' of two [Current] plus the [Next] and [Previous] that lie between. Bézier curves are then generated 'vertically' through these quads; this is the 'QA'. Then, 'horizontal' curve sets are generated through these QA curves using the successive curve points as control points. The points of this final collection of curves essentially constitute the mesh of the Bézier solid. Also see intro.
*/
function BezierSolid(TPA,SectionResolution=30,SetResolution=20,Doughnut=false) =
    let(
        CPA=BezierControlPolygonArray(BezierTangentVectorArray(TPA)),
        Imax=len(CPA)-2,
        QA= [ // Quad array (made from TPS triples).
            for (Idex=[0:Imax])[ // Quad (between two TPS).
                RemoveDoubles(FlattenArray([for(CP=CPA[Idex  ][0])
                    BezierCubic(CP,SectionResolution)])), // Bottom.
                RemoveDoubles(FlattenArray([for(CP=CPA[Idex  ][1])
                    BezierCubic(CP,SectionResolution)])), // Bottom Next.
                RemoveDoubles(FlattenArray([for(CP=CPA[Idex+1][2])
                    BezierCubic(CP,SectionResolution)])), // Top Previous.
                RemoveDoubles(FlattenArray([for(CP=CPA[Idex+1][0])
                    BezierCubic(CP,SectionResolution)])), // Top.
            ],
            if(Doughnut)[ // Closing Quad.
                RemoveDoubles(FlattenArray([for(CP=CPA[Imax+1][0])
                    BezierCubic(CP,SectionResolution)])), // Bottom.
                RemoveDoubles(FlattenArray([for(CP=CPA[Imax+1][1])
                    BezierCubic(CP,SectionResolution)])), // Bottom Next.
                RemoveDoubles(FlattenArray([for(CP=CPA[0     ][2])
                    BezierCubic(CP,SectionResolution)])), // Top Previous.
                RemoveDoubles(FlattenArray([for(CP=CPA[0     ][0])
                    BezierCubic(CP,SectionResolution)])), // Top.
            ]
        ]
    )
    RemoveDoubles( // Remove duplicate cross sections, i.e., [Current] curves.
        FlattenArray([
            for(Jdex=[0:Len(QA)])
                BezierCubic(QA[Jdex],     // Sections through 'quad-curves'.
                /*
                  Set resolution, i.e., how many 'cross-sections' are generated between each TPS can be over-ridden by specification of the SRO (Section Resolution Override) parameter. This is useful to prevent overcrowding between closely spaced TPS. And your GPU probably appreciates it.
                */
                    undef==TPA[Jdex][2]
                    ?   SectionResolution
                    :   TPA[Jdex][2]
                )
            ]
        )
    )
;

