/*
  Take a set of Bézier control polygon sets like this:

    CPS=[
        [
            [0,0],[12,5],[13,17],[10,20]
        ],
        [
            [10,20],[4,26],[10,40],[20,35]
        ]
    ];

  or a single polygon like this:

    CP=[
        [0,0],[10,5],[5,10],[10,15],[5,20],[10,25],[0,30]
    ];

  and display the steps in calculating a 'DeCasteljau' point on the curve,
  or display the entire curve generation in an animation.

  Fraction :    A number between 0 and 1.

  Because the only 'variable' is a fraction, this just screams "Animation!":
  For an animated demonstration of the generation of the entire curve, open
  the animation dialog in OpenSCAD (View ->Animate). CP FPS to 10 and Steps
  to 200, then try this with one of the above Bézier control point sets, or
  get creative and make one yourself..:

  | ShowCastejau(BezierSet,AnimationFraction([2,8,]),$AnimSegs=10);

  Resolution : Number of points calculated on the curve.
*/
module ShowDeCasteljau(CPS,Fraction=1,Resolution=50){

    _CPS=is_num(CPS[0][0]) // Only one Bézier set?
    ?   [CPS]
    :   CPS;

    /* Clip Fraction between 0 and 1. BYNK. */
    _Fraction=Clip(Fraction);

    /* Some colors for the different sets */
    Colors=[RED,OSG,BLU,GRN,PRP,DSG,CYN,MGT,YLW];

    // Stitch curves together into one.
    for (CP=_CPS) {
        Curve=BezierCurve(CP,Resolution);
        // After the entire curve is created [in the animation], it increases in
        // diameter to 'swallow' the control points and make it stand out more.
        Rr=_Fraction==1?0.51:0.3; // _Fraction=1 (we're finished) -> Bigger radius.
        color(SVR)
        /* _Fraction smaller than 1/Resolution means _less than one curve segment_ and will throw an exception. Prevent this with if() condition. */
        if(_Fraction>1/Resolution)ShowCurve(Curve,Rr,_Fraction,true,false);
    }

    for(CP=_CPS){ // Generate an array of de Casteljau points.
        Steps=DeCasteljauSteps(CP,_Fraction);

        for(Jdex=[0:len(Steps)-2]){
            color(Colors[Jdex]) // For each step in the de Casteljau algorithm.
            for(Kdex=[0:len(Steps[Jdex])-2]){
                translate(Steps[Jdex][Kdex])
                sphere(r=0.5); // Show the points.
                // Connect the points.
                CylinderBetween(Steps[Jdex][Kdex],Steps[Jdex][Kdex+1],Radius=0.25);
            }
            color(Colors[Jdex])
            translate(Steps[Jdex][len(Steps[Jdex])-1])
            sphere(r=0.5); // This step's end point.
        }
        color("silver")
        translate(Steps[len(Steps)-1][0])
        sphere(r=0.5); // The Bezier point for this fraction.
    }

}

/*
  Generate a list of the successive de Casteljau algorithm results.
  Useful to see how the de Casteljau algorithm arrives at the desired point.
  Only good for demonstration purposes. Probably. Used by ShowDeCasteljau().

  ControlPoints : An array of Bézier curve control points.
  Fraction      : Fraction of the curve for which to calculate the point.
  _Result        : Algorithm internal parameter.
*/
function DeCasteljauSteps(ControlPoints,Fraction,_Result=[])=
    _Result==[]
    ?   DeCasteljauSteps(ControlPoints,Fraction,[ControlPoints])
    :   (len(ControlPoints)==2)
        ?   concat(_Result,[[Fraction*ControlPoints[1] + (1-Fraction)*
            ControlPoints[0]]])
        :   let(
                Step=[for(Idex=[0:len(ControlPoints)-2])
                    Fraction*ControlPoints[Idex+1]+(1-Fraction)*ControlPoints[Idex]
                ]
            )
            DeCasteljauSteps(Step,Fraction,concat(_Result,[Step]))
;

