/*
  Predominantly used  for shapes and polygons containing circular holes.

  * NOTE: Circle(R), unlike Arc(0,360,R) does NOT return the '360 degree endpoint', i.e., if $fn=4, Arc() returns FIVE vertices (the vertex on the X-axis is the start- and end-point of the arc), whereas Circle will only return FOUR, since it knows that [R, 0, 0] is only needed _once_. Therefore, please use the appropriate function.
*/
function Circle(Radius,3D=false)=
    let(
        Steps=Segments(Radius)
    )
    [
        // The last (360) Step would duplicate the first vertex, so we drop it.
        for(Step=[0:Steps-1])
            let(
                A=Step/Steps*360
            )
            // Only generate z-coordinate if *3D*=true.
            [Radius*cos(A),Radius*sin(A),if(3D)0]
    ]
;

// Pattern is dash-dot pattern in unit line lengths, like [5,10] for 5 unit lines with 10 unit gaps, or [3,2,1,2] &c. The Pattern[0] and Pattern[even] elements are lines, the Pattern[odd] elements are gaps.
module Circle(Radius,Thickness=0.1,Pattern=undef)
{
    if(Pattern==undef){
        // Here, i.e, where the torus is formed, $fa, $fs, and $fn determine curve segment resolution.
        rotate_extrude()
        translate([Radius,0])
        circle(r=Thickness/2,
            // Here, i.e., where the torus cross section is formed, $fc rules, unless it's set to _0_...
            $fn=$fc==0
            ?   Segments(Radius,$fn=0) // $fn=0 here 'disables' $fn in Segments().
            :   $fc
        );
    }else{
        /* Dash-dot or whatever pattern line.. */
        /* NO error checks here, but lots of room for mess-ups. */
        /* We no longer care about *FullEnds* and such, we're just drawing lines for an image or 'drawing' (OpenSCAD ain't no AutoCAD). */
        PatternAngle=ArraySum(Pattern/Radius/Tau*360);
        PatternCount=max(floor(360/PatternAngle),1);
        /* Adjust pattern fit and convert to angles. */
        PatternFactor=360/(PatternAngle*PatternCount);
        Fattern=Pattern*PatternFactor/Radius/Tau*360;
        FatternAngle=ArraySum(Fattern);
        /* Draw the circle. */
        for(Idex=[0:PatternCount-1])
            for(Jdex=[0:Len(Fattern)])
                if(IsEven(Jdex))
                    let(Start=Idex*FatternAngle+(Jdex>0?ArraySum(Fattern,0,Jdex-1):0))
                    Arc(Start,Start+Fattern[Jdex],Radius,Thickness,undef,true);
    }
}

