
//===TROCHOIDS==================================================================

/*
  Trochoid.

  Takes radius dimensions for the fixed and rolling circles, as well as the 'generator'--the latter relative to the rolling circle's center.

  R     Fixed circle radius.
  r     Rolling circle radius, when negative, the hypotrochoid or, if
        *d*=*r*, the hypocycloid is generated.
  d     Radius of the generator from the _center_ of the Rolling circle,
        it lies on the (extended) line from the rolling circle centre to its contact point with the fixed circle when its centre lies on the x-axis.
  Angle Optional, angle around the fixed circle that the rolling circle
        shall make. If left _undef_, the rolling circle will make so many revolutions as is required to bring the generator or 'scribing point' back to the starting position, i.e., the full trochoid will be generated.
  NOTE: $fn vertices are generated on the curve. Specify $fn in the call if
        you want a certain number of vertices.

  TIP:  This routine distinguishes epi- and hypo-trochoids by the
        sign of the rolling circle's radius. For this reason, there are several 'abs()' calls that may not make much sense. They are there to take care of indiscriminate users throwing around '-' signs with gay abandon, and to make some other parameters behave appropriately.

  // Example to get a certain height trochoid; used by clock gears:

    // Trochoid data:
    R=100;  // Fixed circle.
    r=R/2;  // Rolling circle.
    h=5;    // Trochoidal tooth 'height'.

    // Triangle sides:
    A=r;    // Rolling circle radius (actually, rolling circle center to generator vertex, but that's the same...).
    B=R+h;  // Fixed circle center to trochoidal tooth 'tip'.
    C=R+r;  // Rolling circle center to fixed circle center.

    // Three known sides give one angle, in this case, the one opposite side B. This is (R/r) times the Trochoid() *Angle* parameter, so multiply by r/R.
    Angle=acos((Pow2(A)+Pow2(C)-Pow2(B))/(2*A*C))*r/R;

  // End example.
*/
function Trochoid(R,r,d,Angle)=
    let(
        /* The factors. */
        dr=abs(R)+r,    // Don't accept negative *R*.
        f=abs(R)/r+1,   // *f* follows sign of *r*.
        s=abs(r)/r*d,   // *s* is *d* with sign of *r*.
        /* If no generating angle is specified, generate the entire trochoid. This rotates the rolling circle and generator point until the latter returns to its original position. */
        /* Generate vertices. */
        _Angle=(Angle==undef
        ?   360*LCM(R,r)/R
        :   Angle),
        Steps=round(_Angle/360*$fn==0?Segments(r)*R/r:$fn)

    )
    [
        for(Sdex=[0:Steps])
            let(Adex=Sdex/Steps*_Angle)
            [
                dr*cos(Adex)-s*cos(f*Adex),
                dr*sin(Adex)-s*sin(f*Adex)
            ]
    ]
;

/*
  For animations make the call like so:
  ShowTrochoid(R,r,d,DSG,true,CFN=6);
  * Set $fn to the animation 'Steps' number or the curve and 'writer' will not coincide in each frame.

  For non-animations make the call like so:
  ShowTrochoid(R,r,d,DSG,false,CFN=6,$t=1);
  * Set $fn to any value (or omit for default), $t=1 shows the entire curve, $t<1 shows part of the curve (or omit for current value, which is probably _0_, unless...).

  R     scalar    Fixed circle radius.
  r     scalar    Rolling circle radius.
  d     scalar    Length of 'generator'.
  Kolor color     TheGHOUL or OpenSCAD colour.
  Fade  Boolean   If _true_ the curve fades 'with age'.
  Alpha fraction  color alpha value, is applied to ALL curves in the routine.
                  Only here because I needed it for
                  _Demofiles/Img_Roulettes.scad_

*/
module ShowTrochoid(R,r,d,Colors=[OSG,DSG,CPR,GPG],Alpha=1)
{
    CT=0.2; // Curve Thickness.

    Angle=360*LCM(R,r)/(R); // Total angle (rotations) required for a fully developed curve.

    color(Colors[0],Alpha)
    ShowCurve(
        Trochoid(R,r,d)
    ,CT,Frac=$t,Connect=true,Closed=false);

    color(Colors[1],Alpha)
    ShowCurve(
        Circle(R)
    ,CT/2.2,Frac=1,Connect=true,Closed=true);

    color(Colors[2],Alpha)
    rotate([0,0,$t*Angle])
    translate([R+r,0,0])
    ShowCurve(
        Circle(abs(r))
    ,CT/2,Frac=1,Connect=true,Closed=true);

    color(Colors[3],Alpha)
    rotate([0,0,$t*Angle])
    translate([R+r,0,0])
    rotate([0,0,R/r*$t*Angle+(Sign(r)+1)*90]){
        BallCylinderBetween(
            [0,0,0],[d,0,0],Radius=CT/2
            // Use $fc--if set--to construct the curve, i.e., for the curve's elements, or the 'default' number of segments otherwise.
            ,$fn=$fc==0
                ?   Segments(CT/2)
                :   $fc
        );
        translate([d,0,0])
        sphere(
            r=CT/0.7
            // Use $fc--if set--to construct the curve, i.e., for the curve's elements, or the 'default' number of segments otherwise.
            ,$fn=$fc==0
                ?   Segments(CT/2)
                :   $fc
        );
    }

}

//===INVOLUTES==================================================================
/*
  Actually, this should say 'INVOLUTE', singluar, or even better (since there are many variations), 'CIRCLE INVOLUTE'.
*/
/*
  I'm getting lazy; if you don't know what the terms below mean, go to TheGHOUL's documentation on gears, I'm not doing it all twice...

  This routine is somewhat 'tuned' for gear-tooth-profile generation, if you just want a plain-old circle involute, call it with only *BaseRadius* defined.

  BaseRadius,   scalar,     radius of the base (fixed) circle.
  TipRadius,    scalar,     radius of the tip circle.
  StartRadius,  scalar,     radius where the curve starts.
*/
function Involute(BaseRadius,TipRadius,StartRadius)=
    let(
        _TipRadius=TipRadius==undef?BaseRadius*PI:TipRadius,
        TipAngle=InvoluteGenAngle(BaseRadius,_TipRadius),
        StartAngle=StartRadius==undef
        ?   0
        :   InvoluteGenAngle(BaseRadius,StartRadius),
        GenAngle=TipAngle-StartAngle,
        Steps=Sequence(StartAngle,TipAngle,$fa)
    )
    [
        for(Alpha=Steps)
            BaseRadius*[cos(Alpha),-sin(Alpha),0]+
            BaseRadius*Alpha*PI/180*[sin(Alpha),cos(Alpha)]
    ]
;
module Involute(BaseRadius,TipRadius,StartRadius)
{
    // For animations. At the start, when Tip==Base, Involute() yields [[nan,nan]].
    if(TipRadius>BaseRadius)
        ShowCurve(
            Involute(BaseRadius,TipRadius,StartRadius)
        ,Radius=0.1,Frac=1,Connect=true,Closed=false);
}

// Some involute() helpers.

/*
  Generating angle of involute of BaseRadius at Radius ('height').
*/
function InvoluteGenAngle(BaseRadius,Radius)=
    sqrt(Pow2(Radius)-Pow2(BaseRadius))/BaseRadius/PI*180
;

/*
  Including angle ('width') of involute of BaseRadius at Radius.
*/
function InvoluteVertexAngle(BaseRadius,Radius)=
    let(
        IGA=InvoluteGenAngle(BaseRadius,Radius)
    )
    IGA-atan(BaseRadius*IGA*PI/180/BaseRadius)
;

