/*
  Optimal number of segments for a circle or circular arc, based on $fa, $fs, $fn, in a method SIMILAR to OpenSCAD's.

  There will NEVER be fewer segments than 4.
  With this caveat in mind:
  *If $fn is NOT 0, there will be $fn segments.
    Otherwise (i.e., $fn=0),
    *If *Radius* is undefined, the segment NUMBER will be 360/$fa.
    Otherwise (i.e., $fn=0 with a known radius),
      * The segment SIZE will be at least $fs, and the segment NUMBER will be no more than 360/$fa (there's a cut-off radius where the two switch).
      * UNLESS $ff is NOT 0, in which case the segment size will never be larger than $ff, regardless of what $fs and $fa have to say.

  To ensure vertices on the cardinal directions:
  $fn should be a multiple of 4 but (preferably) no more than 128 (or your CPU may not love you). Default is 0 which hands control to $fa and $fs; any value<>0 overrides $fa and $fs.
  $fa should be a factor of 90 and no less than 0.1 (OpenSCAD will complain); default is 12--which is not a factor of 90--so The GHOUL uses 6, which yields 60 segments on a circle, like the minutes on a clock; how nice is that?
  $fs should be a (small) multiple of your printer's resolution (nozzle size) and no less than 0.1 (OpenSCAD will complain); The GHOUL's default is 0.4.
  $ff should not be too small, for the sake of your CPU, default is 0 (i.e. 'off'); make it as large as you can 'stand it', and no smaller than your printer's resolution (because, well, that would be silly).
*/
function Segments(Radius)=
    $fn>0       // $fn overrides EVERYTHING; with great power comes
                // great responsibility.
    ?ceil(max(3,$fn)) // Since $fn is considered an override, we ignore
                // The GHOUL's '4-fold' rule here, and additionally we allow
                // a minimum of _3_ so that we can make equilateral triangles.
    :MCeil(
        max(
            Radius==undef
                // Without a dimensional reference, all we have is angles.
            ?360/$fa
                // $ff is the largest acceptable 'flat' side, and
                // it overrides $fs and $fa.
            :max($ff==0?0:Radius*2*PI/$ff,
                min(
                    // The number of $fs size sections for this radius.
                    Radius*2*PI/$fs
                    // The number of $fa angles per full circle.
                    ,360/$fa
                )
            )
        )
        ,4  // >>> Full circles MUST** be a 4-fold so that there
            // >>> are vertices on the cardinal directions.
            // ** That is to say, according to The GHOUL, i.e., me.
            // The GHOUL's hard minimum is thus 4 segments per full circle.
            // (Seriously, is a square a circle? I feel this should perhaps
            // be at least 8, even 12, certainly not OpenSCAD's vague
            // minimum of 3 or 5, depending on whether $fn is set or not...)
    )
;

/*
  OpenSCAD native algorithm; see $fa, $fn, $fs entry in the manual, OpenSCAD uses the term 'fragments' so I've done the same.
*/
function Fragments(Radius=1)=
    $fn>0.0?max($fn,3):ceil(max(min(Radius*2*PI/$fs,360/$fa),5))
;
