/*
  <GHOUL/Lib/Lib_Animation.scad>

  Hank J. van Leuvensteijn Jr. 2021
  hankjr@hankjr.ca

  The GHOUL (Great Helpful OpenSCAD Unified Library) is set free under the unlicense:

  Anyone is free to copy, modify, publish, use, compile, sell, or distribute this software, either in source code form or as a compiled binary, for any purpose, commercial or non-commercial, and by any means.
*/

// Introduction and TODO =======================================================
// Give square wave a fullness factor?

/*
 * Let's get this out of the way first: OpenSCAD isn't POV-Ray or Blender or
 * Animaker or what-have-you, but sometimes it's just nice to have stuff move,
 * or change color, or zoom in or...
 * Well, that's what the animation view is for, but it's not too user-friendly.
 * Enter: The Animation.scad library... (Far from perfect or finished...)

  OpenSCAD provides the $t special variable which grows in linear fashion from
  0 at the beginning of the animation sequence to 1 at the end. It provides
  therefore a fraction of the entire sequence, which can be used to create
  animations by directly incorporating it in an object's call.

  Say we want to create a 15 second animation sequence. We want a cube that
  becomes visible at 3 seconds and disappears at 7 seconds. We also want it
  to be transparent at the start, fully opaque in the middle at 5 seconds,
  and transparent again just before it disappears:

  | if ($t>0.2&&$t<0.467)
  |     color("red",0.3+0.7*pow(sin(($t-0.2)/(0.467-0.2)*180),2))
  |     cube([1,1,1]);

  It requires a fair understanding of math, and some serious bookkeeping of
  your start and end factors. An additional problem is that when the total
  length of the animation changes, all factors for $t also need to change.
  Hmmm, not much 'parametric' about that. Of course you can do:

  | Length=15;
  | CStart=3;
  | CEnd=7;
  |
  | if ($t>CStart/Length&&$t<CEnd/Length)
  |     color("red",
  |     0.3+0.7*pow(sin(($t-CStart/Length)/(CEnd-CStart)/Length*180),2))
  |     cube([1,1,1]);

  More parametric and kind of 'more better' to speak with Capt. Jack Sparrow,
  but it's still ugly code, and if you have several animated objects it gets
  old very quickly.

  Or you can use this library:

  | $AnimSegs=15; // Total animation lasts 15 seconds.
  | Animate([3,7,7]) color("red",$AnFlash) cube([1,1,1]);

  The vector parameter given to 'Animate()' is an 'AnimVector', a vector
  that describes when an object begins (to appear or change), stops changing,
  and when it ends (disappears).
  Clear and easy.
  'Animate()' provides the special variable '$AnFlash' (and a lot of other ones)
  to it's children() to influence color, position, rotation, you name it.
  Check out 'Hank-Jr-Libraries/DemoFiles/AnimatedThings.scad'
  for a smorgasbord of options.

  * To be able to do this, the library needs to know the length of the total
  * animation sequence, for which it uses the special variable '$AnimSegs'.
  * '$AnimSegs=...;' MUST be declared in the main (top level) file so it is
  * available to all animation routines.

  The Animation view in OpenSCAD (View -> Animate) has two settings:
  Steps and FPS. Let's say you want 15 seconds of animation at 10 Frames Per
  Second [FPS], then you will need 10 FPS x 15 Seconds = 150 Steps.

  In the above example we would want to declare $AnimSegs=15; for one second
  per animation segment.

  What if you need to extend the sequence to 20 seconds instead of 15? Simply
  increase $AnimSegs 20 and Steps to: $AnimSegs*FPS = 20*10 = 200.
  The animations that are already defined will be unaffected in real time
  because each segment still lasts one second.

*/

include<Animation/AnFrac.scad>
include<Animation/Animate.scad>
include<Animation/Animator.scad>
include<Animation/BezierTransitionUp.scad>
include<Animation/BezierTransitionDown.scad>
include<Animation/Cycle.scad>
include<Animation/StringTee.scad>
include<Animation/ViewPort.scad> // Contains a few viewport routines.

/* The functions below are NOT in the Geany configuration files. */
include<Animation/AnimationFxFraction.scad>
include<Animation/AnimationFxSquare.scad>
include<Animation/AnimationFxTriangular.scad>
include<Animation/AnimationFxTwoSines.scad>

