/*
  Perform affine transformation on an array of vertices.

  Vector:   An openSCAD rotate() vector [x,y,z].
  List:     An array of vertices or a single vertex.
  Angle:    Rotation angle with Vector as axis.
  Output2D: Force 2D output, even for 3D input, i.e., crop Z.

  Returns 3D or 2D consistent with input.
*/
function AffineRotate(Vector,List,Angle=undef,Output2D=false)=
    let(
        Matrix=Angle==undef
        ?   RotationMatrix(Vector)
        :   RotationMatrix(Vector,Angle)
    )
    undef==Len(List[0])
    // List is a vertex.
    ?   Len(List)==1||Output2D
        // 2D output.
        ?   CropTuple(ProjectVector(Matrix*HomogeniseVector(List)))
        // 3D output.
        :   PadTuple(ProjectVector(Matrix*HomogeniseVector(List)))
    // List is an array of vertices (we hope).
    :       Len(List[0])==1||Output2D
        // 2D output.
        ?   CropArray([ for(Vertex=List)
                    ProjectVector(Matrix*HomogeniseVector(Vertex))
                ])
        // 3D output.
        :   PadArray([ for(Vertex=List)
                ProjectVector(Matrix*HomogeniseVector(Vertex))
            ])
;

/*
  If you _know_ you're dealing with a 3D list input and return...
  See above for parameter descriptions.
*/
function AffineRotate3D(Vector,List,Angle=undef)=
    let(
        Matrix=Angle==undef
        ?   RotationMatrix(Vector)
        :   RotationMatrix(Vector,Angle)
    )
    undef==Len(List[0])
    ?   ProjectVector(Matrix*concat(List,1))
    :   [ for(Vertex=List)
            ProjectVector(Matrix*concat(Vertex,1))
        ]
;

