/*
  Affine Rotation Matrix
  Input is rotation axis (as a vector) and the angle in degrees, similar to
  OpenSCAD 'rotate(a=...,v=[..,..,..])'
  If no angle is specified, the vector is treated as a vector of Euler angles
  in degrees similar to OpenSCAD 'rotate([x,y,z])'.
*/
function RotationMatrix(Vector=[0,0,0],Angle=undef)=
    Angle==undef
    /* 'Vector' contains Euler rotation angles. */
    ?   [
            [cos(Vector.z), -sin(Vector.z), 0, 0],
            [sin(Vector.z),  cos(Vector.z), 0, 0],
            [            0,              0, 1, 0],
            [            0,              0, 0, 1]
        ]*
        [
            [ cos(Vector.y), 0, sin(Vector.y), 0],
            [             0, 1,             0, 0],
            [-sin(Vector.y), 0, cos(Vector.y), 0],
            [             0, 0,             0, 1]
        ]*
        [
            [1,             0,              0, 0],
            [0, cos(Vector.x), -sin(Vector.x), 0],
            [0, sin(Vector.x),  cos(Vector.x), 0],
            [0,             0,              0, 1]
        ]
    /* 'Vector' is an actual vector and is the rotation axis. */
    :   let(
            _Vector=UnitVector(Vector),
            L=_Vector.x,
            M=_Vector.y,
            N=_Vector.z,
            O=1-cos(Angle)
        )
        [
            [
                L*L*O+cos(Angle),
                M*L*O-N*sin(Angle),
                N*L*O+M*sin(Angle),
                0
            ],
            [
                L*M*O+N*sin(Angle),
                M*M*O+cos(Angle),
                N*M*O-L*sin(Angle),
                0
            ],
            [
                L*N*O-M*sin(Angle),
                M*N*O+L*sin(Angle),
                N*N*O+cos(Angle),
                0
            ],
            [
                0,0,0,1
            ]
        ]
;
