// check out Steiner points
/*
    Cover an array of polygons with a polyhedral mesh.
    The 2D polygons are transecting the desired polyhedron in 3D space
    and MUST have their vertices ordered in a *clockwise fashion* when looking at the 'bottom' of the finished polyhedron.

    [
      [   [x11,y11,z11],[x12,y12,z12]...[x1P,y1P,z1P]   ]  each polygon has the same length of 'P' elements
      [   [...],        [...],       ...[...]           ]  each element is a vertex [x,y,z]
      [   [xN1,yN1,zN1],[xN2,yN2,zN2]...[xNP,yNP,zNP]   ]  there are 'N' polygons in the set
    ]

    Endcaps :   Boolean, if true, the mesh is like a tube capped on both ends.
    Donut  :   Boolean, if true, the mesh is like a doughnut, ends connected.

*/
function CoverMesh(Polygons, Endcaps=true, Donut=false)=

    let(
        Foo=Alert(["Donut takes precedence over Endcaps in CoverMesh.scad."],0,Endcaps&&Donut),

        PolygonVertices = len(Polygons[0]),    // Number of vertices per polygon.
        NumberPolygons = len(Polygons),        // Number of polygons.
        NumberVertices = NumberPolygons*PolygonVertices,

        Vertices = [                      // Flatten the array of polygons to one
            for (Polygon = Polygons)      // dimension i.e: a simple array of
                for (Vertex = Polygon)    // vertices (a vertex cloud).
                    Vertex
        ],

        // For each vertex of each polygon along the polyhedron, fill it's
        // quadspace in 'forward direction' (right hand rule) along the polyhedron.
        Hull = [
            for(Polygon = [1:NumberPolygons-1])
                for (Vertex = [0 : PolygonVertices-1])
                    let(Space=QuadSpace(Polygon,Vertex,PolygonVertices,NumberVertices))
                        for(Face=Space) Face
        ],

        // Close the first polygon, in a clockwise direction.
        StartCap    = [[
            for(Index = [0 : PolygonVertices-1])
                Index
        ]],

        // Close the last polygon, this happens in 'reverse direction'
        // (high to low index) to maintain the clockwise direction of face
        // definition when looking at the face from outside the polyhedron.
        EndCap      = [[
            for(Index = [(PolygonVertices*NumberPolygons)-1 : -1 : PolygonVertices*(NumberPolygons-1)])
                Index
        ]],

        Closure     = [
            for (Vertex = [0 : PolygonVertices-1])
                let(Space=QuadSpace(NumberPolygons,Vertex,PolygonVertices,NumberVertices))
                    for(Face=Space) Face
        ],

        // Concatenate the applicable face arrays (Donut y/n, Endcaps y/n).
        Faces=
            Donut
            ?   concat(Hull, Closure)
            :   Endcaps
                ?   concat(StartCap, Hull, EndCap)
                :   Hull
    )

    [Vertices,Faces]
;

module CoverMesh(Polygons, Endcaps=true, Donut=false, Convexity=10) {

    Mesh=CoverMesh(Polygons, Endcaps, Donut);

    // create the polyhedron
    polyhedron(convexity = Convexity, points = Mesh[0], faces = Mesh[1] );
}

module ShowMesh(Polygons,Endcaps=true,Donut=false,Begin=0,End=1,Mono=undef,Alpha=1) {

    Mesh=CoverMesh(Polygons, Endcaps, Donut);
    //Colors=[RED,CYN,DRD,GRN,MGT,DGN,BLU,YLW,DBL];
    Colors=Mono==undef?[RED,BLU]:Mono;
    LC=len(Colors);

    for(Fdex=[Len(Mesh[1])*Clip(Begin):Len(Mesh[1])*Clip(End)]){
        color(Colors[Fdex%LC],Alpha)
        polyhedron(
            points=Mesh[0],
            faces=[Mesh[1][Fdex]]
        );
    }
}

