/*
  TODO: leader dimension/radius dimension. Maybe.
*/
/*
  OpenSCAD won't be competing with AutoCad any time soon, just get yourself LibreCAD if you have ambitions in that direction, but this little routine here is jolly useful for illustrations and documentation.

  * Although it's pretty robust, when you try a silly thing like using an aligned dimension for a (near) vertical, dimension, you may get silly results. Just don't be silly; OpenSCAD isn't AutoCad.

  First, Second:    Vertices, points to be dimensioned.
  Type:             String, alignment of the dimension, "H", "V", or "A",
                    for Horizontal, Vertical, or Aligned.
  TSize, TSpacing:  Integers, text properties.
  TMove:            Vector, incremental text position.
  Decimals:         Integer, number of decimals to display.
  Spacing:          Distance between object and dimension line.
  Extension:        Distance the extension line go past the dimension line.
  Gap:              Distance between the object and the extension line.
  Direction:        "L", "C", or "R", placing of text, Left, Center, or Right.
  Location:         "A" or "B", location of dimension line above or below
                    the object.
  Thickness:        Line thickness.
  Marks:           "A or "C" for arrows or circles.
  MSize:            Mark size.
  Outside:          Boolean, marks are placed outside the extension
                    lines if 'true'.
  Rotation:         Not yet implemented, rotation of the text around the
                    dimension line, to align it with the $vpr vector in order to increase readability.

*/
module Dimension(First,Second,Type="H",Text="",TSize=1,TSpacing=1,TMove=[0,0,0],Decimals=2,Spacing=2,Extension=0.5,Gap=0.25,Direction="R",Location="B",Thickness=0.05,Marks="A",MSize=0.3,Outside=false,Rotation=0){
    /* Internal Mark module. */
    module Mark(){
        if(Marks=="A"){
            rotate([0,Outside?90:-90,0])
            translate([0,0,-Thickness/2])
            cylinder(r1=0,r2=MSize/2,h=MSize);
        }else if(Marks=="C"){
            sphere(r=MSize/2.5);
        }
    }

    /* How many parameters do you need? This'll do fine here.. */
    DimColor=DSG;
    TextColor=SVR;
    /* Some definitions and helpers. */
    Org=[0,0,0];
    AbBe=Location=="A"?1:-1;
    RiLe=Direction=="L"?-1:Direction=="R"?1:0;
    View=$vpr;
    /* If both Z-coordinates are 0, we're not 3D. */
    3D=!((First.z+Second.z)==0);

    /* The Vector between the dimension points needs to be translated to the origin, then rotated so we can dimension horizontally along the X-axis; later we rotate and translate back into position. Let's find that Vector.*/
    /* Raw vector. */
    _Vector=Second-First;
//    Length=Modulus(_Vector);
    zRot=atan2(_Vector.y,_Vector.x);
    /* 3D dimension needs to be rotated into Z=0 plane. */
    ZRot=3D?180*round(zRot/180)-zRot:0;
    XRot=3D?-90:0;
    __Vector=AffineRotate([XRot,0,0],
                AffineRotate([0,0,ZRot],
                    _Vector));
    /* Direction of the vector in the Z=0 plane. */
    Dirn=ATan2(__Vector.x,__Vector.y);
    /* Align for vertical and aligned dimensions, the raw vector is only suitable for horizontal dimensions. */
    VRot=Type=="H" // Horizontal needs no additional rotation.
    ?   0
    :   Type=="V" // Vertical needs 90 degree rotation.
        ?   -90
        /* Aligned is a bit more fussy; text for dimension lines from -45 to 135 degrees go 'right-side-up', in the other 180 degree sector, they go 'upside-down'. */
        :   ((Dirn+45)%360)<180
            ?   -Dirn
            :   -Dirn+180;
    /* Relying on the funky atan2() followed by two affine transforms, combined with the relatively poor resolution in OpenSCAD, we're going to end up wit not-quite-zeros where we expect to see a 0; better force them into obedience... */
    Vector=ThoughtZero(AffineRotate([0,0,VRot],__Vector));
    Length=Modulus(Vector);

    /* Some helper dimensions. We take the center of the vector as our reference point, that way we can simply multiply our offsets with 1 or -1 for high-low and right-left. */
    Middle=Vector/2;
    Center=[abs(Middle.x),abs(Middle.y),0];
    /* Format text. */
    String=Text!=""
    ?   Text
    :   Type=="A"
        ?   SetDecimals(Length,Decimals)
        :   SetDecimals(abs(Vector.x),Decimals);
    StringLength=len(String)*TSize*TSpacing;
    /* Dimension line length compensation for 'outside' marks. */
    OS=Outside
    ?   MSize
    :   0;

    /* The dimension is drawn with *First* on the origin, so translate the whole thing there. */
    translate(First)
    /* Reverse vector rotations. */
    rotate([0,0,-ZRot])
    rotate([-XRot,0,0])
    rotate([0,0,-VRot])
    union(){
        color(DimColor){
            /* Place marks. */
            translate([min(0,Vector.x),Middle.y+(Center.y+Spacing)*AbBe,0])
            rotate([0,180,0])
            Mark();
            translate([max(0,Vector.x),Middle.y+(Center.y+Spacing)*AbBe,0])
            Mark();
            /* The Dimension line. */
            BallCylinderBetween(
                [
                  Middle.x-Center.x+min(0,(Extension+OS+StringLength)*RiLe),
                  Middle.y+(Center.y+Spacing)*AbBe,
                  0
                ],
                [
                  Middle.x+Center.x+max(0,(Extension+OS+StringLength)*RiLe),
                  Middle.y+(Center.y+Spacing)*AbBe,
                  0
                ],
            Thickness/2);
            /* The left extension line. */
            BallCylinderBetween(
                [0,Gap*AbBe,0],
                [0,Middle.y+(Center.y+Spacing+Extension)*AbBe,0],
            Thickness/2);
            /* The right extension line. */
            BallCylinderBetween(
                [Vector.x,Vector.y+Gap*AbBe,0],
                [Vector.x,Middle.y+(Center.y+Spacing+Extension)*AbBe,0],
            Thickness/2);
        }
        /* The text. */
        color(TextColor)
        translate(
            /* TMove is the user's 'fine tuning' parameter. */
            TMove+
            [
              Middle.x+(Center.x+Extension+OS+StringLength/2)*RiLe,
              TSize+Middle.y+(Center.y+Spacing)*AbBe,
              0
            ]
        )
        SimpleText(String,TSize,Thickness=0.1,Center=true,Font="Noto Mono",Spacing=TSpacing);
    }
}

