// Includes ====================================================================
include<TimingBeltData.scad>

// Notes and Conventions =======================================================

// Functions ===================================================================

// Modules =====================================================================

/*
 * Generates a GT belt tooth polygon, either straight, or conforming to the
 * appropriate PCD (Pitch Circle Diameter).
 * The generated tooth is placed centered on the positive x-axis with the
 * center of it's PCD on the origin so only linear extrusion along the z-axis
 * and copying into a circular array is required to complete a cutting body for
 * a pulley.
 *
 * Teeth    : Number of belt teeth of the pulley (determines shape).
 * Teeth>0  : A tooth shaped to the PCD calculated for a pulley
 *            of the given number of teeth.
 * Teeth=0  : A straight tooth (like a rack).
 *
 * Clearance provides more height (IBelt) outside of the pitch line when
 * making a cutting body for example in a belt tightener
 */
module GTTooth(Teeth=0,Type="GT3-2",Clearance=0){

    Pitch=BeltData(Type)[0];
    TipRadius=BeltData(Type)[1];
    FlankRadius=BeltData(Type)[2];
    RootRadius=BeltData(Type)[3];
    HBelt=BeltData(Type)[4];        // Total height of belt and tooth
    HTooth=BeltData(Type)[5];       // Height of tooth
    IBelt=BeltData(Type)[6];        // Thickness (height) of belt
    PLD=BeltData(Type)[7];          // Pitch Line location
    FlankRadiusOffsetY=BeltData(Type)[8]; // The FlankRadii are offset to +/- Y

    PCR=Teeth*Pitch/PI/2;
    AngularPitch=360/Teeth;
    Dstep=1;    // Resolution of generated points on circles in degrees.
    Lstep=0.05; // Resolution of generated points on straight lines in units.

    // Tip radius
    Adx=PCR-HTooth-PLD+TipRadius;
    Aarc=90-atan((HTooth-TipRadius)/FlankRadiusOffsetY);
    Apoints = [ for (i=[0:Dstep:Aarc]) let(X=Adx-cos(i)*
        TipRadius,Y=sin(i)*TipRadius) [X,Y] ];

    // Flank radius
    Bdx=PCR-PLD;
    Bdy=-FlankRadiusOffsetY;
    Bstart=90-atan((HTooth-TipRadius)/FlankRadiusOffsetY);
    Barc=90-Bstart-asin(RootRadius/(FlankRadius+RootRadius));
    Bpoints = [ for (i=[0:Dstep:Barc]) let(X=Bdx-cos(i+Bstart)*
        FlankRadius,Y=Bdy+sin(i+Bstart)*FlankRadius) [X,Y] ];

    // Root radius
    Cdx=PCR-RootRadius-PLD;
    Cdy=cos(asin(RootRadius/(FlankRadius+RootRadius)))*
        (FlankRadius+RootRadius)-FlankRadiusOffsetY;
    Cstart=asin(RootRadius/(FlankRadius+RootRadius));
    Carc=90-Cstart;
    Cpoints = [ for (i=[0:Dstep:Carc]) let(X=Cdx+cos(-90+i+Cstart)*
        RootRadius,Y=Cdy+sin(-90+i+Cstart)*RootRadius) [X,Y] ];

    // Belt groove (receives pulley tooth)
    DXstart=PCR-PLD;
    DYstart=Cpoints[len(Cpoints)-1][1];
    Dlen=Pitch/2-DYstart+Smidge;
    Dstep1=Dlen/round(Dlen/Lstep);
    Dpoints= [ for (i=[0:Dstep1:Dlen]) let(X=DXstart,Y=i+DYstart) [X,Y] ];

    // Belt 'cut' (i.e. the end face of the single tooth belt section)
    EXstart=PCR-PLD;
    EYstart=Pitch/2*1.01;
    Elen=HBelt-HTooth+Clearance;
    Epoints= [ for (i=[0:Lstep:Elen]) let(X=i+EXstart,Y=EYstart) [X,Y] ];

    // Outside belt surface
    FXstart=PCR+IBelt-PLD+Clearance;
    FYstart=Pitch/2*1.01;
    Flen=FYstart;
    Fpoints= [ for (i=[0:Lstep:Flen]) let(X=FXstart,Y=FYstart-i) [X,Y] ];

    // Gpoints is half a belt tooth
    Gpoints=concat(Apoints,Bpoints,Cpoints,Dpoints,Epoints,Fpoints);
    // Hpoints is mirror image of Gpoints (the 'other' half)
    Hpoints= [for (i=[len(Gpoints)-1:-1:0])
        let(X=Gpoints[i][0],Y=-Gpoints[i][1]) [X,Y] ];

    // Both halves united into a single straight tooth
    Ipoints=concat(Gpoints,Hpoints);


    if(Teeth==0){ // Straight tooth.
        polygon(Ipoints);
    }else{        // Bend the tooth onto the pitch circle along it's pitch line.
        Jpoints= [for (i=[0:len(Ipoints)-1])let(
            X=Ipoints[i][0]*cos(Ipoints[i][1]*180/PI/PCR),
            Y=Ipoints[i][0]*sin(Ipoints[i][1]*180/PI/PCR)) [X,Y] ];
        polygon(Jpoints);
    }

}

/*
  Assign and publish PCD.
  Use this to find required dimensions when making a belt cutout in a solid etc.
*/
function GTPCD(Teeth=0,Type="GT3-2")=
    let(
    Pitch=BeltData(Type)[0]
    )
    Teeth*Pitch/PI;
;

/*
  Generate a simple belt pulley.
  Shaft hole and flanges need to be added, a hub if required etc.
  Teeth : Number of teeth on the pulley.
  Width : Width of the belt surface of the pulley.
  Type  : Belt type used, see TimingBeltData.scad.
*/
module GTPulley(Teeth=20,Width=6,Type="GT3-2"){

    Pitch=BeltData(Type)[0];
    PCD=Teeth*Pitch/PI;
    Echo(["PCD = ",PCD]);

    difference(){
        cylinder(r=PCD/2,h=Width,center=true);

        linear_extrude(height=Width+Cut,center=true,convexity=10)
        union(){
            for(Index=[0:Teeth])
                rotate([0,0,Index*360/Teeth])
                GTTooth(Teeth=Teeth,Type=Type,Clearance=0);
        }
    }
}

// EOF =========================================================================
