/*
  Chainwheel.
  Assumes oval chain links with semi-circular ends.
    Length      : Dimension, largest outside dimension of the chain links.
    Width       : Dimension, smallest outside dimension of the chain links.
    Gauge       : Dimension, link wire diameter.
    Links       : Integer, number of links to fit on wheel circumference.
    Flange      : Dimension, thickness of wheel (side) flange.
    ID          : Dimension, center hole or bore.
    Shake       : Dimension, tolerance, play, extra room for links in recess.
*/
module ChainWheel(Length=8,Width=4,Gauge=1,Links=20,Flange=2,ID=6,Shake=0)
{
// LENGTHS
    /* Pitch is the largest internal dimension of the links, i.e., the effective link length. */
    Pitch=Length-Gauge*2;

    /* See image in this directory for clarification of this trig stuff... */
    /* Half Flat Link. */
    OA=(Pitch+Gauge)/2;
    /* OA minus link end radius. */
    OC=OA-(Width-Gauge)/2;
    /* Half Edge Link. */
    AB=(Pitch-Gauge)/2;
    /* Alpha, angle included by half Flat Link and half Edge Link. */
    Alpha=360/Links;

// RADII
    /* Radius to center Edge link. */
    EC=(OA+AB*cos(Alpha))/sin(Alpha);

    /* Pitch circle radius, not really, because the pitch as defined above becomes fuzzy when you wrap. Pitch/2/sin(180/Links) is close, but no cigar. */
    PCR=sqrt(Pow2(AB)+Pow2(EC));
    /* Radius to center Flat link. */
    FC=sqrt(Pow2(PCR)-Pow2(OA));
    /* The OD... */
    OD=PCR+Gauge;

    Print(["OD of chain wheel is ",OD*2," mm, wind-out is ",Pitch*Links," mm or ",InchFraction(Pitch*Links/25.4)," inch per turn."]);

    ER=EC-Width/2; // Radius to inside (groove) Edge link.
    FR=FC-Gauge/2; // Radius to inside ('bed') Flat link. *********
    IR=ID/2; // Inside radius, ID, you know, center bore, hole...

    Print(["FR=",FR]);

// DIMENSIONS
    T=(Width+Flange)/2+Shake; // Total wheel thickness.
    W=Width/2+Shake; // Width of Flat link 'bed'.
    G=Gauge/2+Shake; // Width of Edge link groove.

// RESOLUTIONS
    Wra=360/MRound(Pitch*Links/$fs,4); // Wheel resolution angle.
    Lra=360/MRound(Width*PI/$fs,4); // Link resolution angle.

// ANGLES
    // Alpha, see above.
    Beta=acos(FC/PCR)+asin(G/PCR); // Angle to Flat link extreme, i.e., where the Flat link bed transfers to Edge link slot; the 'chain-stop'. >> However... We should stop short of this because the stop doesn't reach the center of the link width, since there is a slot running down the entire wheel to accommodate the 'edge links'. It's only a smidge though, perhaps less...
    Gamma=atan(OC/FC);
    Sigma=asin(Shake/PCR); // Angle of Shake at PCR.
    Epsilon=asin(G/W);

    // Create a half-section; contains a half bed and a half slot.
    // The helper function is called each time, and by changing the various diametral and thickness/width parameters between calls, different cross-section polygons are generated that ultimately shape the slots and reliefs which receive the chain links in the chainwheel.
    Half=[
        for(Angle=Interval(Wra/4,Wra,Gamma,Wra/4))
            // Flat link bed up to Beta.
            _CWCrossSection(OD,FR/cos(Angle),ER,IR,T,W,G,Angle),
        for(Angle=Interval(Lra,Lra,90-Epsilon,Lra/4))
            // Radiused bed.
            _CWCrossSection(OD,sqrt(Pow2(FC)+Pow2(OC+W*sin(Angle)))*FR/FC,ER,IR,T,W*cos(Angle),G,atan((OC+W*sin(Angle))/FC)),
        for(Angle=Interval(Beta+Wra,Wra,Alpha-Wra/4,Wra/4))
            // Edge link slot.
            _CWCrossSection(OD,OD,ER,IR,T,G,G,Angle)
    ];

//Print(["RRR=",sin(atan((OC+W*sin(90-Epsilon))/FC))*FC]);
//Print(["BBB=",Beta]);

    // Mirror-copy two halves together to create one section. Avoids 'duplicate' centre polygon.
    One=concat(
        [
            for(Idex=[Len(Half):-1:1])
                AffineScale([1,-1,1],Half[Idex])
        ],
        Half
    );

    if(IsOdd(Links))
    Stop(["Links must be an even integer."]);
    else
        // For trouble-shooting, or just because.
        if(true) // <<<<<< Change the Boolean to switch behaviour.
        // A. Render the whole chainwheel.
        CoverMesh(
            [for(Idex=[0:Links/2-1])
                for(PGon=One)
                    AffineRotate([0,0,1],PGon,Idex*Alpha*2)
            ],
            false,true
        );
        else
        // B. Render only one section.
        CoverMesh(One
        );

}

/*
  Helper for ChainWheel() above; creates a single polygon, forming a radial cross-section of the chainwheel, like the cut you'd make in a circular cake.
  You'll have to figure this out yourself. Not sorry.
*/
function _CWCrossSection(OD,FR,ER,IR,T,W,G,Angle)=
    let(
        /* Prevent co-location of vertices, which CGAL doesn't like. Reduce it if your printer is _that_ accurate ;-). */
        DD=0.001
    )
    AffineRotate([0,0,1],
        [
            [IR,0,T],
            [OD,0,T],
            [OD,0,W],
            [FR,0,W],    // In the slot, W=G, DD prevents co-location
            [FR-DD,0,G], // of these two vertices, the same happens below.
            [ER,0,G],
            [ER,0,-G],
            [FR-DD,0,-G],
            [FR,0,-W],
            [OD,0,-W],
            [OD,0,-T],
            [IR,0,-T],
        ]
        ,Angle
    )
;


// =============================================================================


module SquareChainWheel(Length=8,Width=4,Gauge=1,Links=20,Flange=2,ID=6,Shake=0)
{
    Pitch=Length-Gauge*2; // Pitch is the largest internal dimension of the links, i.e., the effective link length.

    $fn=MRound(Pitch*Links/0.5,4);
    Print(["ChainWheel $fn= ",$fn]);
    FA=360/$fn;
    Print(["ChainWheel FA= ",FA]);

    // See image in this directory for clarification of this trig stuff...
    OA=(Pitch+Gauge)/2; // half Flat Link.
    AB=(Pitch-Gauge)/2; // half Edge Link.
    Alpha=360/Links; // Alpha, angle included by half Flat Link and half Edge Link.
    EC=(OA+AB*cos(Alpha))/sin(Alpha); // Radius to center Edge link.

/*
    The quick and dirty is close, but not correct, by about 1 in 10 000 or so.
    PCR=Pitch/2/sin(180/Links);
*/
    PCR=sqrt(Pow2(AB)+Pow2(EC)); // Pitch circle radius, not really, because the pitch as defined above becomes fuzzy when you wrap.
    FC=sqrt(Pow2(PCR)-Pow2(OA)); // Radius to center Flat link.
    OD=PCR+Gauge; // The OD...

    Print(["OD of chain wheel is ",OD*2," mm, wind-out is ",Pitch*Links," mm per turn."]);

    ER=EC-Width/2; // Radius to inside (groove) Edge link.
    FR=FC-Gauge/2; // Radius to inside ('bed') Flat link. *********
    IR=ID/2; // Inside radius, ID, you know, center bore, hole...
    T=(Width+Flange+Shake)/2; // Total wheel thickness.
    W=(Width+Shake)/2; // Width of Flat link 'bed'.
    G=(Gauge+Shake)/2; // Width of Edge link groove.

    Beta=acos(FC/PCR)+asin(W/2/PCR); // Angle to Flat link extreme, i.e., where the Flat link bed transfers to Edge link slot; the 'chain-stop'.
    Sigma=asin(Shake/PCR); // Angle of Shake at PCR.

    // Skip nearest $fn step to Beta if within 'Frc' part. 0<Frc<0.5
    Frc=max(2*Sigma/360*$fn,0.1); // If Beta falls within Frc of a $f step, the step is dropped (essentially, Beta becomes that step).
    Pre=floor(Beta/360*$fn-Frc); // Pre-Beta $fn step.
    End=Alpha/360*$fn;
    // Post has to be smaller than End to prevent issues at Edge link slot, see below.
    Post=min(End,ceil(Beta/360*$fn+Frc)); // Post-Beta $fn step.

    ExMid=End%1==0
    ?   1
    :   0; // Flag, if End falls exactly on a $fn location, we need to drop this polygon when we mirror-copy the first half sector into the second half to create the whole 'One', so we don't get coinciding polygons.

    // Create a half-section; contains a half bed and a half slot.
    // The helper function is called each time, and by changing the various diametral and thickness/width parameters between calls, different cross-section polygons are generated that ultimately shape the slots and reliefs which receive the chain links in the chainwheel.
    Half=[
        for(Idex=[0:Pre])
        let(Angle=Idex*360/$fn)
            // Flat link bed up to Beta.
            _CWCrossSection(OD,FR/cos(Angle),ER,IR,T,W,G,Angle),
            // Beta.
            _CWCrossSection(OD,FR/cos(Beta),ER,IR,T,W,G,Beta),
            // Beta + Sigma (Shake).
            _CWCrossSection(OD,OD,ER,IR,T,G,G,Beta+Sigma),
        for(Idex=[Post:End]) // Post/End trouble-point is here.
            // Edge link slot.
            _CWCrossSection(OD,OD,ER,IR,T,G,G,Idex*360/$fn),
    ];

    // Mirror-copy two halves together to create one section.
    One=concat(
        [
            for(Idex=[len(Half)-1-ExMid:-1:1])
                AffineScale([1,-1,1],Half[Idex])
        ],
        Half
    );

    // For trouble-shooting, or just because.
    if(true) // <<<<<< Change the Boolean to switch behaviour.
    // A. Render the whole chainwheel.
    CoverMesh(
        [for(Idex=[0:Links/2-1])
            for(PGon=One)
                AffineRotate([0,0,1],Idex*Alpha*2,PGon)
        ],
        false,true
    );
    else
    // B. Render only one section.
    CoverMesh(One
    );
}


