/*
  Gears in clocks are a horror show, a dog's breakfast, compared to 'proper' involute gears. They are so, because, unlike involute gears, clock gears--running by necessity without lubrication--are made to avoid contact between the 'wheel' teeth and 'leaves' (pinion teeth) before the 'line of centers'. In clocks, the gear or 'wheel' generally drives, and the (driven) pinions are very small because very large ratios must be achieved in--preferably--low numbers of gears.
  Luckily, we're dealing with--for all intents and purposes--invariable, continuous loads and very low powers*.
  Since we're only interested in the driving (i.e. retreating) side of the line of action (after the line of centers) we use only the addendum of the driving gear (the wheel) and the dedendum of the driven gear (the pinion). As a generating circle**, we use half the pinion's pitch diameter so the dedendums will be straight radial flanks***, and only the wheel's teeth's addendums need any significant 'construction' as the wheel's roots as well as the pinion's leaves' roots and tips are all simple circular curves.

  *     A grandfather clock weight of 10 Lbs falling 60 inches in 7 days for example, only works at a rate of 0.005 ft*lbf/min, or 1.5 e-7 horsepower, it's teensy.
  If you prefer the devilishly deviant French units, a 5 kg weight falling 1.5 m in 7 days works at a rate of 0.00012 watt.
  Given such small powers, it's amazing that all those gears keep going, don't forget, it's also driving the pendulum. All in all, it should be clear that friction is the devil's tool to make your clock misbehave, or fail to run entirely, hence the requirement for the abhorrent epicyclic gears, with lots of 'shake', or backlash, to avoid any chance of contact before the line of centers.
  **    If you're lost at this point, I suggest you do some homework first, perhaps consult Wikipedia on the subject of epicyclic gears.
  ***   See 'Tusi couple' on Wikepedia, and run ShowTrochoid(4,2,2); in animation view to see this.
*/
/*
    Generate an epicyclic clock gear (see above comments).
    Wheel         : Integer, number of teeth on the wheel.
    Pinion        : Integer, number of leaves on the pinion.
                    Both *Wheel* and *Pinion* are required because the wheel's teeth are based on the same generating circle, based on the pinion's PCD.
    DP            : Rational, diametral pitch. This is the number of teeth
                    per unit diameter of the pitch circle, i.e., if DP=2 and
                    the pitch circle has a diameter of 20 units, the gear will have 2x20=40 teeth. A larger DP gives smaller teeth.
    Shake         : Fraction, the clockmaker's word for 'play' or 'slack',
                    removes material at the flanks and tips. Some shake is required to prevent binding, which would stop the clock. This _also_ provides root clearance, since material is removed at the tips.
                    * Generally, it is sufficient to give some, say 0.05 shake to the wheel only, and leave the pinion full size. Also, usually, set RootClearance to zero on both.
    Tip           : Dimension, flat or cropped tooth-tip width.
    RootClearance : Dimension, removes material at the roots to provide
                    clearance with the tips of teeth and leaves, 0 is usually fine for both, if some shake is given to the wheel.
    ID            : Dimension, internal diameter, shaft hole, bore.
    String        : Reference for console output.
    Thickness     : Dimension, thickness of the gear, i.e., axial dimension.
    Color         : Color of the gear in the viewer, in hex notation or
                    TheGHOUL defined color names.
    Mesh:         : Real, the gear is rotated by *Mesh* number of pitches
                    to maintain mesh in the viewer. Generally this is necessary if the gear has an uneven number of teeth.
    WShake        : Pinion ONLY, optional parameter, specifying the shake
                    applied to the meshing wheel. If given, the root depth of the pinion leaves can be minimised, thus optimising strength.
*/
/*
    Epicyclic gears are made by rolling a _generating_ circle on a pitch circle and tracing the track of a point on it's circumference. Ish...
    For them to mesh, the pinion leaves and wheel teeth must be generated by a circle of identical diameter, being half the pinion pitch circle diameter.
*/

module ClockWheel(Wheel, Pinion, DP=1, Shake=0.05, Tip=0.1, RootClearance=0, ID=0, String="Gear", Thickness=1, Color=OSG, Mesh=0){

    C=ClockWheel(Wheel, Pinion, DP, Shake, Tip, RootClearance, ID, String);
    color(Color)
    linear_extrude(Thickness)
    rotate([0,0,Mesh*180/Wheel])
    polygon(C.x,C.y);

}

module ClockWheel_M(Wheel, Pinion, M=1, Shake=0.05, Tip=0.1, RootClearance=0, ID=0, String="Gear", Thickness=1, Color=OSG, Mesh=0){

    C=ClockWheel(Wheel, Pinion, 1/M, Shake, Tip, RootClearance, ID, String);
    color(Color)
    linear_extrude(Thickness)
    rotate([0,0,Mesh*180/Wheel])
    polygon(C.x,C.y);

}

module ClockPinion(Wheel, Pinion, DP=1, Shake=0, Tip=0.1, RootClearance=0, ID=0, String="Gear", WShake=0, Thickness=1, Color=OSG, Mesh=0){

    C=ClockPinion(Wheel, Pinion, DP, Shake, Tip, RootClearance, ID, String, WShake);
    color(Color)
    linear_extrude(Thickness)
    rotate([0,0,Mesh*360/Pinion])
    polygon(C.x,C.y);

}

module ClockPinion_M(Wheel, Pinion, M=1, Shake=0, Tip=0.1, RootClearance=0, ID=0, String="Gear", WShake=0, Thickness=1, Color=OSG, Mesh=0){

    C=ClockPinion(Wheel, Pinion, 1/M, Shake, Tip, RootClearance, ID, String, WShake);
    color(Color)
    linear_extrude(Thickness)
    rotate([0,0,Mesh*180/Pinion])
    polygon(C.x,C.y);

}

// EDIT--No longer used
module UnShake(Teeth, DP, Shake){

    rotate([0,0,Shake*180/PI/Teeth*DP])
    children();

}

/*
  This is the routine that generates the polygon points for the epicycloid teeth of the wheel.
*/
function ClockWheel(Wheel, Pinion, DP=1, Shake=0.05, Tip=0.1, RootClearance=0, ID=0, String="Gear", Thickness=undef, Color=undef, Mesh=undef)=
    let(
        // Pitch radius.
        WheelPR=Wheel/DP/2,
        // Radius of the generating circle (which traces the tooth profile), using half the pinion PCR gives straight line flanks on the pinion leaves.
        GenR=Pinion/DP/2/2,
        // Half tooth angle. i.e., one quarter of circular the pitch angle minus shift for shake.
        TAngle=(1/2-Shake)*180/Wheel,
        // The rest is half the gap (root) angle.
        WRAngle=180/Wheel-TAngle,
        // Root radius.
        RootR=(WheelPR-RootClearance)*sin(WRAngle/2)*2,
        // Root-radius centre.
        RootC=WheelPR-RootClearance,
        // Generating arc and tooth height.
        TD=_GenData(WheelPR,GenR,1/1000/DP,Tip,TAngle,0,1/10/DP,GenR),
        GenAngle=TD[0],
        // Optimal number of tooth generating steps, no less than 4. The arc used is the generating angle (how far the generating circle rolls around the gear pitch circle) times the ratio of the gear pitch circle over the generating circle, i.e., the rotation of the generating circle. This is not equal to the length of the tooth flank arc of course, but it's a decent approximation.
        GSteps=max(4,ceil(Segments(GenR)*GenAngle/360*WheelPR/GenR)),
        // Giving the angles a 'sinusoid' distribution results in a more even spacing of the vertices, especially near the tooth tip. The old formula is Angles=[for(Idex=[1:GSteps])Idex/GSteps*GenAngle]. Note we drop the first [0] vertex; that one's ridiculously close to the first root vertex.
        Angles=[for(Idex=[1:GSteps])sin(Idex/GSteps*90)*GenAngle],
        Points=
        concat(
            AffineRotate([0,0,180/Wheel]
                ,ArrayAdd(
                    Arc(182,270-WRAngle,RootR)
                    ,[RootC,0]
                )
            )
            ,
            [
                for(Angle=Angles)
                    [   // The tracing point is the point on the generating circle that touches the gear pitch circle at the start of the curve generation.
                        // X-Coord of generating circle center.
                        cos(TAngle-Angle)*(WheelPR+GenR)
                        // X-Coord of tracing point on generating circle.
                        -cos(TAngle-Angle*(1+WheelPR/GenR))*GenR
                        // Y-Coord of generating circle center.
                        ,sin(TAngle-Angle)*(WheelPR+GenR)
                        // Y-Coord of tracing point on generating circle.
                        -sin(TAngle-Angle*(1+WheelPR/GenR))*GenR
                    ]
            ]
        ),
        // Combine separate tooth copies to make a gear.
        Gear=FlattenArray(
            [ for(Idex=[0:Wheel-1])
                // Rotate a copy for each tooth, and rotate the whole gear so the first tooth's pitch-point is on the X-axis; this is to enable reliable meshing in the viewer--in case that's something you may need...
                AffineRotate([0,0,Idex/Wheel*360-TAngle],
                    // Mirror the half-tooth and combine to make a whole tooth.
                    concat(ScaleArray(Points,[1,-1]),ReverseArray(Points))
                )
            ]
        ),
        Hole=ID>0
        ?   Circle(ID/2)
        :   [],
        Bar=Print(["\n",String," pitch radius is ",WheelPR,"."
                    ,"\nTip radius = ",WheelPR+TD[1]
        ])
)
    [
        concat(Gear,Hole),
        Hole==[]
        ?   [
                [ for(I=[0:Len(Gear)]) I ]
            ]
        :   [
                [ for(I=[0:Len(Gear)]) I ],
                [ for(J=[len(Gear):len(Gear)+Len(Hole)]) J ]
            ]
    ]
;


/*
  This is the routine that generates the polygon points for the hypocycloid leaves of the pinion.
*/
function ClockPinion(Wheel, Pinion, DP=1, Shake=0, Tip=0.1, RootClearance=0, ID=0, String="Gear", WShake=0, Thickness=undef, Color=undef, Mesh=undef)=
    let(
        // Pitch radius.
        WheelPR=Wheel/DP/2,
        PinionPR=Pinion/DP/2,
        // Radius of the generating circle (which traces the tooth profile).
        GenR=PinionPR/2,
        // Angular pitch.
        PitchAngle=360/Pinion,
        // Half tooth angle. i.e., one quarter of circular pitch angle minus shift for shake.
        TAngle=(1/2-WShake)*180/Wheel,//shake
        LAngle=(1/2-Shake)*180/Pinion,
        PRAngle=180/Pinion-LAngle,
        // Tip radius.
        TipR=PinionPR*sin(LAngle/2)*2,
        // Tip centre.
        TipC=PinionPR,
        // Generating arc and tooth height for _0_ tip.
        TD=_GenData(WheelPR,GenR,1/1000/DP,0,TAngle,0,1/10/DP,GenR),
        // Pinion flank Wheel tip exit angle (from generating circle centre).
        Alpha=180-acos((Pow2(WheelPR+GenR)+Pow2(GenR)-Pow2(WheelPR+TD[1]))/2/(WheelPR+GenR)/GenR),
        // Root radius center based on 'lowest' flank contact point or 'exit point' is exit point to pinion centre, divided by cos(rootangle), however, multiplied by the same cos() adds safety compensating for root curvature being stronger than wheel OD curvature. Valid for pinions below 8 leaves--as far as I've seen.
        DaA=sqrt(Pow2(GenR*(1-cos(Alpha)))+Pow2(GenR*sin(Alpha)))*cos(PRAngle)-RootClearance,
        // Root radius center based on wheel tip (fully developed 'sharp' tip; no flat), Valid for pinions over 7 leaves--as far as I've seen.
        DaB=(PinionPR-TD[1])*(1+sin(PRAngle)/(1-sin(PRAngle))),
        // Pick the smallest one (cut-off appears to be at 7-8 leaves).
        RootC=min(DaA,DaB),
        // Root radius.
        RootR=RootC*sin(PRAngle),
        // One side (root and tip rounding) of the leaf.
        Points=
        concat(
            // Root arc.
            AffineRotate([0,0,180/Pinion]
                ,ArrayAdd(
                    Arc(182,270-PRAngle,RootR)
                    ,[RootC,0]
                )
            )
            ,
            // Rounding (tip).
            ArrayAdd(Arc(90+LAngle,1,TipR),[TipC,0])
        ),
        // Combine leaf copies to make a pinion.
        Gear=FlattenArray(
        [ for(Idex=[0:Pinion-1])
            // Rotate a copy for each leaf.
            AffineRotate([0,0,Idex/Pinion*360-LAngle],
                // Mirror the half-leaf and combine to make a whole leaf.
                concat(ScaleArray(Points,[1,-1]),ReverseArray(Points))
            )
        ]
        ),
        // Center hole, if specified.
        Hole=ID>0
        ?   Circle(ID/2)
        :   [],
        // Some user info.
        Foo=Print([ "\n",String," pitch radius is ",PinionPR,"."
                    ,"\nTip radius = ",TipC+TipR,"."
                    ,"\nCore diameter is ",(RootC-RootR)*2,"."
//                    ,"\nDaA = ",DaA,"."
//                    ,"\nDaB = ",DaB,"."
            ])
    )
    [
        // Vertices.
        concat(Gear,Hole),
        // Paths.
        Hole==[]
        ?   [
                [ for(I=[0:Len(Gear)]) I ]
            ]
        :   [
                [ for(I=[0:Len(Gear)]) I ],
                [ for(J=[len(Gear):len(Gear)+Len(Hole)]) J ]
            ]
    ]
;

/*
    Support routine for ClockWheel() and ClockPinion().
    Returns [Angle, Dimension]. Angle is the angle of the arc through which the generating circle rolls over the pitch circle. Dimension is the tooth height outside the pitch circle.

    My trig skills are too rusty, and I'm too lazy to brush them up, so I just approached it incrementally. I'm not even sure if there's another way to do this, and this is _plenty_ accurate, and certainly fast enough. Engineering is full of empirical stuff, so I figured this would do. Also, Sheldon would hate it...

    * If you've got the skills to solve
    * sin(A-B)*(R+r)-sin(A-B*(1+R/r))*r=X
    * for B, let me know at hankjr@hankjr.ca...
*/
function _GenData(WheelPR,GenR,Resolution,Tip,TAngle,Angle,Step,Result)=
    Result>(Tip/2+Resolution)
    /* Still approaching, do it again... */
    ?   _GenData(WheelPR,GenR,Resolution,Tip,TAngle,Angle+Step,Step,
            sin(TAngle-(Angle+Step))*(WheelPR+GenR)
            -sin(TAngle-(Angle+Step)*(1+WheelPR/GenR))*GenR
        )
    :   Result<max(Tip/2,0)
        /* Overshoot. Go back 3/4 step and quarter step-size. */
        ?   _GenData(WheelPR,GenR,Resolution,Tip,TAngle,Angle-Step*3/4,Step/4,
                sin(TAngle-(Angle-Step*3/4))*(WheelPR+GenR)
                -sin(TAngle-(Angle-Step*3/4)*(1+WheelPR/GenR))*GenR
            )
        /* Result! (We're between Tip/2 and Tip/2+Resolution)*/
        :   [Angle,
                cos(TAngle-Angle)*(WheelPR+GenR)
                -cos(TAngle-Angle*(1+WheelPR/GenR))*GenR-WheelPR
            ]
;

