/*
  EOT WHEEL AND CALCULATION METHODS ###################################### START
*/
/*
  Equation Of Time wheel. (A cam, really.)
  EOT minutes (+/-) times factor F are superimposed to 'Zero radius' R0.
  This cam can be used to move an indicator on an EOT scale. EOT is the amount of minutes a sundial appears to be fast or slow compared to standard time. EOT minimum is approx. -14'6" (slow) around 11 Feb and maximum is approx. 16'33" (fast) around 3 Nov.
  The equation of time has zeros near 15 Apr, 13 Jun, 1 Sep, and 25 Dec.
  It's a nice 'complication' but--for a number of reasons that, like EOT itself, really don't matter a fig--pretty much useless.

  The EOT wheel is generated with New-Year on the positive x-axis, you could diffrence() a notch or a pinhole somewhere as a reference.

  See also: https://en.wikipedia.org/wiki/Equation_of_time

  RMin :    Radial distance to the minimum dT, near Feb 14.
  RMax :    Radial distance to the maximum dT, near Nov 1.
  ID :      Internal diameter.
  Ind :

  */
function EOTWheel(RMin=20, RMax=40, ID=6, Ind=0.1)=

    let(
        Max=EOT(305), // Nov. 1st
        Min=EOT(45), // Feb 14th
        Ftr=(RMax-RMin)/(Max-Min),
        RZero=RMax-Max*Ftr,

        EOT=[
            for (Day=[1:365])
            PolarToVector([RZero+Ftr*EOT(Day),360/365.24*Day])
        ],
        Bore=(undef==ID||0==ID)
        ?   []
        :   Circle(ID/2),
        Dmd=Ind>0
        ?   ArrayAdd(Diamond(EOT[1].x*Ind,EOT[1].x*Ind/2)[0],[EOT[1].x*(0.8-Ind),0])
        :   []
        ,Data=Print([
            "EOTWheel \n* R Zero value = ",RZero
            ,"\n* At Jan. 1st, EOT is ",EOT(1)," minutes at R = ",Modulus(EOT[1])
            ,"\n* The EOT minimum is ",EOT(45)," minutes at Feb. 14th at R = ",RMin
            ,"\n* The EOT maximum is ",EOT(305)," minutes at Nov. 1st at R = ",RMax
            ,"\n* Cam K-factor is ",Ftr," mm radial movement per minute indicated."
        ])
    )
    [
        concat(EOT,Bore,Dmd),
        [
            RangeToTuple([0:Len(EOT)]),
            Bore==[]?Bore:RangeToTuple([len(EOT):len(EOT)+Len(Bore)]),
            Dmd==[]?Dmd:RangeToTuple([len(EOT)+len(Bore):len(EOT)+len(Bore)+Len(Dmd)])
        ]
    ]
;

module EOTWheel(RMin=20,RMax=40,ID=6,Ind=0.1,T=1)
{
    EOT=EOTWheel(RMin,RMax,ID,Ind);
    linear_extrude(T)
    polygon(EOT[0],EOT[1]);
}

// Same as above, but with radial lines indicating the months.
module EOTWheel_MR(RMin=20,RMax=40,ID=6,Ind=0.1,T=1)
{
    EOT=EOTWheel(RMin,RMax,ID,Ind);
    linear_extrude(T)
    polygon(EOT[0],EOT[1]);

    // Month radials.
    Rads=[for(Mon=[0:11])ArraySum(MonthDays,0,Mon)];
    echo(Rads);
    for(Angle=Rads)
    color(RED)
    translate([0,0,1])
    BallCylinderBetween(
        PolarToVector([ID/2+1,Angle/365.24*360]),
        PolarToVector([Modulus(EOT[0][Angle-1])-1,Angle/365.24*360]),
    Radius=0.3,Ends="Both");
}

/*
  EOT calculation from NOAA Global Monitoring Division.
*/
function EOT(Day=1)=
    let(
        C=360/365.24*(Day-1-12/24),
        EOT=229.18*(0.000075+0.001868*cos(C)-0.032077*sin(C)-0.014615*cos(2*C)-0.040849*sin(2*C))
    )
    EOT
;

/*
  EOT calculation from
  https://en.wikipedia.org/wiki/Equation_of_time#Alternative_calculation
*/
function EOT1(Day=1)=
    let(
        n=360/365.24,
        A=(Day+9)*n,
        B=A+0.0167*360/PI*sin((Day-3)*n),
        C=(A-atan(tan(B)/cos(23.44)))/180,
        EOT=720*(C-round(C))
    )
    EOT
;

/*
  EOT calculation from
  https://en.wikipedia.org/wiki/Equation_of_time#Final_calculation

  seems to be out of phase with both EOT() and EOT2()...
*/
function EOT2(Day=1,Year=2024)=
    let(
        // replace 6.240... with 6.360... for better agreement with the other two routines...
        D=(6.24004077+0.01720197*(365.24*(Year-2000)+Day))*360/2/PI,
        EOT=-7.659*sin(D)-9.863*sin(2*D+3.5932)

    )
    EOT
;

/*
  EOT WHEEL AND CALCULATION METHODS ######################################## END
*/
