/*
  Generate a knurled cylinder.
  Radius    : Outside radius of the knurled cylinder (top of the knurling).
  Height    : Height of the knurled cylinder.
  Size      : Size of the knurling 'pyramid' sides.
  Peak      : height of the knurling 'pyramids'.
*/
module KnurledCylinder(Radius=5,Height=5,Size=2,Peak=0.8){

    RCount=round(PI*Radius*2/Size)*2; // The nearest whole number of pyramids
    //                                   that fit the circumference.
    RStep=360/RCount;

    HCount=max(2,round(Height/Size)*2); // The nearest whole number of pyramids
    //                                     that fit the height.
    HStep=Height/HCount;

    NP=Peak/2;
    R=Radius-NP; // The average radius (halfway up the pyramids).

    Cloud=[ // generate a vertex cloud (points at each pyramid corner and top).
        for(i=[0:HCount]) [ // One circle of points ('profile') for every 'Size'
            //                 along the length.
            for(j=[0:RCount-1]) // Points alternate between top and corner of
                //                 the pyramid by adding or subtracting 'NP'
                //                 from or to 'R' when 'j' is odd (NegOneIfOdd)
                //                 and this is reversed for the next profile
                //                 when 'i' is odd so pyramids in successive
                //                 profiles are offset.
                [i*HStep,(NegOneIfOdd(i)*NegOneIfOdd(j)*NP+R)*cos(j*RStep),
                (NegOneIfOdd(i)*NegOneIfOdd(j)*NP+R)*sin(j*RStep)],
        ],
    ];

    KnurledMesh(); // Cover the cloud of points in a skin.

    /*
      Cover a vertex cloud  generated by 'Knurled'.
      Any other vertex cloud will NOT be properly covered, this is due to the
      alternating offset of the pyramid faces.
    */
    module KnurledMesh(){

        P=len(Cloud);     // Number of profiles (rings around the cylinder).
        PV=len(Cloud[0]); // Number of vertices (points) per profile.
        NV=P*PV;          // Total number of vertices.

        function OctoSpace(P,V,O) = [ // Generate a half set of triangles (faces)
        //                               around each pyramid peak, on the side
        //                               of the origin only.
        // P = current profile.
        // V = current vertex.
        // O = 0 if P is odd, 1 if P is even, to account for the offset of the
        // pyramids on alternating profiles along the cylinder.
        //
        [(V+O)%PV+PV*P,V-1+O+PV*P,(V+O)%PV+PV*(P-1)],
        [V-1+O+PV*(P-1),(V+O)%PV+PV*(P-1),V-1+O+PV*P],
        [(V+1+O)%PV+PV*P,(V+O)%PV+PV*P,(V+O)%PV+PV*(P-1)],
        [(V+O)%PV+PV*(P-1),(V+1+O)%PV+PV*(P-1),(V+1+O)%PV+PV*P]
        ];

        Vertices = [                 // Flatten the 2-D cloud into a 1-D array.
            for (Profile = Cloud)
                for (Vertex = Profile)
                    Vertex
        ];

        Faces = [          // Generate faces between the vertices of the cloud.
            for (Profile = [1:P-1])       // For each profile (slice).
                for (Vertex = [1:2:PV-1]) // For each vertex (point).
                    for(Faces=OctoSpace(Profile,Vertex,ZeroIfOdd(Profile)))
                        Faces  // Call 'OctoSpace' and flatten the array to 1-D.
        ];

        Startcap = [      // Generate faces (triangles) for the origin end-cap.
            [ for(i=[PV-1:-1:0])
                i
            ]
        ];

        Endcap = [        // Generate faces (triangles) for the end end-cap.
            [ for(i=[NV-PV:NV-1])
                i
            ]
        ];

        _Faces = concat(Faces,Startcap,Endcap); // Attach the end-caps.

    // Generate the polyhedron.
    rotate([0,-90,0])
    polyhedron(convexity = 10, points = Vertices, faces = _Faces);

    }

}

// EOF =========================================================================
