/*
    +++++ ++++  +   +   +   +  +++  +++++   +++++  +++    ++++  +++ +++++
      +   +   + +   +   ++  + +   +   +       +   +   +   +   +  +  +
      +   ++++   + +    + + + +   +   +       +   +   +   +   +  +  +++
      +   +   +   +     +  ++ +   +   +       +   +   +   +   +  +  +
      +   +   +   +     +   +  +++    +       +    +++    ++++  +++ +++++

    If you're even THINKING of printing 'load bearing' threads:
    >> Get the hell out of here!
    If you're sick enough to even CONSIDER printing NPT threads for an application in ANY gas piping, or piping subjected to EVEN THE SLIGHTEST PRESSURE:
    >> You need SERIOUS help; have yourself committed!
    Have you EVER witnessed 'just a garden hose' that blew apart? Do you have kids or pets?
    * Water pressure can be up to 100 PSI in some municipalities, although 50 PSI is more common. A garden hose Internal Diameter of 3/4 inch is quite normal, this makes for a 40+ pound-force trying to tear your coupling apart, or 20+ pounds in lower pressure municipalities. If it succeeds, would you like to get slapped in the face with 40+ pounds? Didn't think so. -- These numbers are valid for a 'dead-ended' hose.
    >>  Do NOT even THINK about using printed threads for ANY pressure applications. Not even in the garden. You sick f#ck.

    For all others; please be aware of the limitations of printed objects. Do NOT rely on the mechanical strength or integrity of printed objects in ANY application where failure of that object presents ANY KIND OF RISK.

    There, you have been warned, I wash my hands in innocence.
*/

/* Lectori salutem.

  This is 'TheGHOUL/Lib/Parts/Thread.scad'; it creates threads of any cross-section, continuous or interrupted, tapered or straight, left- or right-handed, external or internal, with and without support surfaces, but *always with EXTREME precision*.

  Also see the companion library in the same directory, 'ThreadData.scad'.

  Don't you detest poorly commented code? Why publish a piece of code if the user has to dig through it all to understand what you're doing for so long, they could practically have written it themself? This code is *profusely* commented for your pleasure, understanding and ease of adaptation; there's nothing left to figure out, I've bared all my 'secrets'.

  I (Hank van Leuvensteijn Jr.) have done my best to make everything clear; 90% of this file is comments...

  A word to the wise, however: If you *really* want to understand threads, you'll have to do more than just read the comments in this library; start with 'https://en.wikipedia.org/wiki/Screw_thread' and take it from there. I've assumed a *basic* understanding. If you don't know the difference between 'Lead' and 'Pitch', or worse, you think they're the same thing: Please leave, and get informed first.

  This library leaves little to be desired when it comes to thread generation, but it doesn't chew your food for you: It generates threads. You're still responsible for generating appropriate, bolt shanks, leading cones (if desired) and whatever else you can attach a thread to**, but it builds just about any (useful) thread you can imagine, accurately, and it enables you to modify them in every sensible and potentially useful way.

  **Have a look at 'SimpleThread()' if you're in a hurry; it does studs, bolts, nuts...

  NOTE: for an internal/external thread set to mesh, all that is required is that one of them be rotated 180 degrees around the Z-axis. This holds true regardless of Pitch, Lead or Multi-Start values.

  This library should have reached you with a number of demo-files in the /TheGHOUL/DemoFiles/... directory. Make a backup of them and play around with the parameters to get a good feel for what's possible. I can especially recommend 'ThreadDemoCalls.scad' for a good look at different threads.

  For readability --and ease of removal-- there are no 'inline' comments; all comments are on their own line, with the following convention:
*/

// Either a TODO or a temporary comment, not meant for the reader, does not contain information pertaining to the code.

/* A regular short-ish or one-liner comment. Yes, using multi-line delimiters, because you never know... */

/*
  A true multi line comment that spans several lines. Not this one of course; this is an example of the format...
*/

/* Is it done? Is it PERFECT? No, Gandalf... A coder's work is never done... Let's rephrase that: A code addict's work is never done. */

//##############################################################################
//#####      ###    ###     ####    ############################################
//#######  ####  ##  ##  ##  ##  ##  ###########################################
//#######  ####  ##  ##  ##  ##  ##  ###########################################
//#######  #####    ###     ####    ###### THAT THERE, THAT'S A BIG TODO. ######
//##############################################################################

// TODO 'sink' profile into supporting surface instead of RiRo scaling?

/*
  This is the thread module. it generates thread helices and their cylindrical support surfaces. For internal threads, it creates cutting bodies to be differemce()d from an object, when 'Support=true'.

  NOTE: The thread helix starts at the positive x-axis, for a different start angle, simply rotate the whole thread afterwards: rotate([0, 0, start_angle]).

  A further NOTE: This module has an *insane* number of parameters. That's the price you pay for full-down-to-the-bone accurate control over the generation of --frankly-- perfect threads. It's a small price to pay IMNSHO, but as always in life: If you want to do something (nearly) perfectly, you must understand it (nearly) perfectly first; you have been forewarned...

  A final note to the trepidatious: It's not that bad. Here's the call for a lovely bit of M5 thread, 10 mm long: Thread("MetricTread",5,10); Like UNC? Here it is: Thread("UNC", 1/4, 1); That's an inch of 1/4 thread for you... Do read the comment on the 'ConversionFactor' before you start mixing metric and imperial threads!

    ThreadType,         string,
                        name of the desired thread profile. Thread profiles are all described in 'TheGHOUL/Lib/Parts/ThreadData.scad'.
    NominalDiameter,    scalar or string,
                        the nominal diameter or a diameter identifier of the external threads, like '#4'. It is the same for external and internal threads, for standardised threads, this is an *identifier*, not necessarily an actual dimension. For non-standard threads, it _is_ the O/D of the external threads.
    Length,             scalar,
                        desired length of the thread.
    External,           Boolean,
                        true for external thread.
    HigbeeIn, -Out,     scalars,
                        degrees of turn over which to apply run-in and/or -out, like a Higbee cut. Default, is 'HigbeeIn=30', 'HigbeeOut=0',
    Higbee2D              Boolean,
                        the Higbee cut will be '2D', i.e, in the radial as well as the axial direction, just try it out.
    HigbeeSkew          scalar, [0:1],
                        the Higbee cut will be skewed axially towards the end of the thread for even better/smoother engagement, only works when 'Higbee2D=true' because skewing the full-width profile axially would just make a mess...
    Turns,              scalar,
                        number of turns of the thread, includes run-in and -out. Compensate for run-in and -out in the call if a specific number of turns of full thread cross section are required. If both 'Length' and 'Turns' are specified, 'Turns' takes prevalence.
    Pitch,              scalar,
                        width of a single thread, also axial travel per turn of a standard single thread. (Standard here means that lead is equal to a single pitch; if you don't know the difference between lead and pitch: GTS.)
    ThreadAngle,        scalar,
                        included angle between thread profile flanks.
    Height,             scalar,
                        full height of thread profile.
    TaperAngle,         scalar,
                        included taper angle of the thread.
    LeftHanded,         Boolean,
                        true for left-handed threads.
    MultiStart,         scalar,
                        multiple thread helixes, like in many food container closures.
    RootClearance,      scalar,
                        clearance at the root of non-standard threads like 'SquareThread'. This does not apply to standard threads as theirs is calculated by the profile generator in the 'ThreadData' library. Positive is less material is more clearance.
    Clearance,          scalar,
                        NOTE: not influenced by ConversionFactor,
                        it is an absolute in the design's NATIVE UNITS.
                        a change in thread size, usually postive in order to provide more actual (positive) clearance, flank contact correction (see below) is available for this.
                        NOTE: 1. This is applied to the thread AND root-radius
                                 of "standard" threads, but NOT to the 'roots' of square threads and such, which have 'root clearance' for that.
                              2. Negative 'Clearance' increases the profile
                                 in radial and axial direction and may (read: probably will) cause interference.
                        NOTE: There is also a built in, production equipment (your 3D printer) dependent, clearance called '_EQ_Allowance', it is set in TheGHOUL/Lib/Definitions/Constats.scad. A notice is given at run-time if _EQ_Allowance is 0.
    FlankContact,       switch [<0,0,>0],
                        rotate the finished thread polyhedron to preserve contact / maintain lock-up point with the mating threads when additional Clearance has been applied.
                        [0]  No preservation.
                        [<0] CCW direction contact preservation.
                        [>0]  CW direction contact preservation.
    Interrupted,        array,
                        interrupted thread parameter array consisting of:
                        [
                            ThreadAngle,
                            GapAngle,
                            HigbeeAngle,
                        ]
                        Default is not interrupted: [0,0,0].
    ConversionFactor,   scalar,
                        thread profiles are written with 'implicit units', i.e. a 1/2" UNC thread has a major diameter of 0.5, not '12.7 mm' so when you're designing in mm and you want to use a UNC bolt, you must specify a ConversionFactor of 25.4 or your thread will have an OD of 0.5 mm.
    Support,            Boolean,
                        generate the support surface for the thread.
    FN                  integer, should be a multiple of 4, resolution of
                        approximations of radii (number of line segments per full circle) generated by the thread-generators in ThreadData.scad. Keep it sensible, these are small radii, 12 or 16 is lots for most threads that aren't knuckle-threads. Larger values can VERY quickly lead to slow rendering speeds.
    _Quiet              internal flag for recursive calls.
    ItemReference       Name of object that the threads are for to enable
                        the user to distinguish individual threads in the console output.
*/

/*
  Generally a call would look like this:

    Thread("MetricThread",10,25);

  But it can get *much* more complex...
*/
module Thread(ThreadType,NominalDiameter,Length=undef,External=true,Turns=undef,Pitch=undef,Lead=undef,ThreadAngle=undef,Height=undef,TaperAngle=undef,LeftHanded=false,MultiStart=1,HigbeeIn=30,HigbeeOut=0,Higbee2D=false,HigbeeSkew=0,RootClearance=undef,Clearance=0,FlankContact=0,Interrupted=[0,0,0],ConversionFactor=1,Support=false,FN=12,_Quiet=false,ItemReference=""){

    /* Get the thread profile (form) from the ThreadData.scad library. Some of the passed parameters will be redundant depending on the thread type, others are only used for non-standard types.
    ThreadData.scad has NO error traps, however, it does ignore Pitch, Height and ThreadAngle input for standardised threads and issues an alert. */
    ThreadData=ThreadData(ThreadType,NominalDiameter,Pitch,ThreadAngle,Height,TaperAngle,Clearance,RootClearance,ConversionFactor);//,$fn=FN);

/* For debugging only. ($Debug is set in Config.scad) show the thread profile as a polygon at the origin. */
if($Debug)color(RED,0.5)translate([ThreadData[5]/2,0,-Tad])linear_extrude(Smidge)polygon(External?ThreadData[0]:ThreadData[1]);

    /*
    First off Error Traps; There's no way to replace an engineering degree with a few error traps. You *must* have a bit of an idea of what's useful and reasonable in threads to use this module. If you don't, remedy that. There are lots of parameters here; I don't even want to guess at the number of possible permutations of them that would break this routine.

    If you don't *know* what you're doing, you *don't* know what you're doing.

    Check this out:
    https://en.wikipedia.org/wiki/Four_stages_of_competence
    */

    /* Either Turns or Length *must* be defined, if both are specified, 'Turns' has precedence. */
    if(undef==Turns&&undef==Length)Stop(["Either 'Turns' or 'Length' *must* be defined."]);
    /* Check that NominalDiameter is in the Pitch dictionary. */
    if(undef==ThreadData[3])Stop(["The lookup table does not have an entry for Nominal Diameter ",NominalDiameter,", you must add it in TheGHOUL/Lib/Parts/ThreadData.scad."]);

    /* Alerts and friendly reminders go here, '_Quiet' is used to avoid publishing these for a second time during the recursive call when 'Support'=true. */
    Alert(["ConversionFactor is set to ",ConversionFactor,"."],0,!_Quiet&&ConversionFactor!=1);

    /* (Re-)Set some parameters... */
    _PitchRadius=ThreadData[5]/2;
    /* $fn replacement. */
    _FN=Segments(_PitchRadius);
    /* 'Pitch' has been processed in ThreadData.scad */
    _Pitch=ThreadData[3];
    /* Lead smaller than pitch is not possible. Lead larger than pitch happens in some food-closures, and rarely elsewhere. */
    __Lead=Lead==undef
    ?   _Pitch
    :   Lead>_Pitch
        ?   Lead
        :   _Pitch;
    /* The lead, taking multi-start into account. */
    _Lead=__Lead*MultiStart;
    /* The height of the *thread profile*, not the thread helix. */
    _Height=ThreadData[4];
    /* Thread profile flank angle is specified as the included angle, we need only half of that. */
    _FlankAngle=ThreadData[10]/2;
    /* Taper is specified as the included angle, we need only half of that. */
    _TaperAngle=ThreadData[11]/2;
    _EQ_Allowance=ThreadData[12];
    /* It seems that CGAL doesn't like it when *Turns* isn't divisible by 1/$fn, i.e. when the end of the thread-helix falls in-between two 'circle-divisions', hence the MRound()-ing used here. What happens is:

    ERROR: CGAL error in CGAL_Nef_polyhedron3(): CGAL ERROR: assertion violation! Expr: ss_circle.has_on(sv_prev->point()) File: /usr/include/CGAL/Nef_3/polygon_mesh_to_nef_3.h Line: 265

    I don't know how (or if) $fn is used in the rendering process, or why this would be an issue, but it does seem somewhat logical to me, so this doesn't really feel like a 'hack' to me, since our resolution is effectively 1/$fn anyway.*/
    _Turns=Turns!=undef
        /* 'Turns' is defined; use it. */
        ?   MRound(Turns,1/_FN)
        /* Use 'Length' to calculate '_Turns', -'_Pitch' is in the calculation because we get a height of one pitch before we even start moving along the helix. */
        :   MRound((Length-_Pitch)/_Lead,1/_FN);
    /* Grab the correct 2D profile, and pad length to 3D. */
    __Profile=PadArray(External?ThreadData[0]:ThreadData[1]);
    /* To maintain clockwise definition of the polyhedron faces, the order of the profile vertices needs to be reversed for left hand thread as -- in a manner of speaking -- the profile order is 'reversed' in the polyhedron and what would have been the 'previous' polygon in the vertex cloud is now the 'next', which causes the mesh generator (which expects 'right-hand-rule' profiles; see comments in QuadSpace.scad) to define counter-clockwise faces, however the endcaps and support surface are still defined clockwise and the mix is a no-no and makes for non-manifoldness, so we 'flip' the profile and everything is 'back in order' again. Pfew. If that doesn't make sense to you; it's no biggie, sleep on it. */
    _Profile=LeftHanded
    ?   ReverseArray(__Profile)
    :   __Profile;
    /* Number of profiles to generate for the thread. */
    Steps=_Turns*_FN;
    /* Turn a boolean into +/-1. so we can use it in our calculations. */
    _Handedness=LeftHanded
    ?   -1
    :   1;
    /* When additional Clearance is provided, flank contact on either the near side of the external thread (tightened) or the far side ('loosened') can be maintained with the 'FlankContact' flag; 1 for the former, -1 for the latter condition, regardless of 'Handedness'.
    NOTE: although 'Clearance' creates equal space, either side of the profile, in case of tapered threads, the rotation needed to re-establish contact is *different* for the two conditions because the profile travels along an inclined path. Have some fun and do the trig.*/
    _ClearanceContactCorrection=FlankContact==0
    ?   0
    :   FlankContact>0
        ?   Clearance/cos(_FlankAngle-_TaperAngle)/_Lead*360*_Handedness
        :   -Clearance/cos(_FlankAngle+_TaperAngle)/_Lead*360*_Handedness;

    /* Support surface. */
    /* The radius at the start (bottom or Z=0) of the thread.*/
    R1= _PitchRadius
        +(External
        ?   ThreadData[0][len(ThreadData[0])-2].x
            -ThreadData[7]*(1/cos(_TaperAngle)-1)
        :   ThreadData[1][1].x
            +ThreadData[6]*(1/cos(_TaperAngle)-1)
        );
    /* The height of the supporting surface. */
    H= _Lead*_Turns+_Pitch;
    /* The radius at the end (top or Z=H) of the thread, compensated for taper (of course), most of the above comments apply also. */
    R2= R1-sin(_TaperAngle)*H/cos(_TaperAngle);

    /* Let's put that in a pretty string so we can publish it. */
    SupSurf=str(
        External==true
        ?   "External "
        :   "Internal ",
        "thread support surface",
        ItemReference==""
        ?   ": "
        :   str(" for ",ItemReference,": "),
        _TaperAngle==0
        ?   "cylinder(r="
        :   "cylinder(r1=",
        R1,
        _TaperAngle==0
        ?   ""
        :   " ,r2=",
        _TaperAngle==0
        ?   ""
        :   R2,
        ", h=",H,"); "
    );

    /* We're using a tiny Fudge-Factor (Overlap) here. To make sure that the thread-roots actually lie *within* the support surface (which is what we are doing next), we give the support surface a _very fine_ overlap. Frogshair. So fine, you can't see it. */
    Overlap=FrogsHair;
    if(Support&&!External){
        /* We're making a cutting body for internal thread so we difference() cut the thread from the support cylinder. The user can then take what we made to difference() cut the thread out of whatever body it's going into. */
        difference(){
            cylinder(r1=R1-Overlap,r2=R2-Overlap,h=H);
            /* We need the exact same call except this time with 'Support=false' and 'Quiet=true'. */
            Thread(ThreadType,NominalDiameter,Length,External,_Turns,Pitch,Lead,ThreadAngle,Height,TaperAngle,LeftHanded,MultiStart,HigbeeIn,HigbeeOut,Higbee2D,HigbeeSkew,RootClearance,Clearance,FlankContact,Interrupted,ConversionFactor,Support=false,_Quiet=true,ItemReference=ItemReference);
        }
    }else if(Support){
        /* We're making external thread with a support cylinder */
        union(){
            cylinder(r1=R1+Overlap,r2=R2+Overlap,h=H);
            /* We need the exact same call except this time with 'Support=false' and 'Quiet=true'. */
            Thread(ThreadType,NominalDiameter,Length,External,Turns,Pitch,Lead,ThreadAngle,Height,TaperAngle,LeftHanded,MultiStart,HigbeeIn,HigbeeOut,Higbee2D,HigbeeSkew,RootClearance,Clearance,FlankContact,Interrupted,ConversionFactor,Support=false,_Quiet=true,ItemReference=ItemReference);
        }
    }else{
         /* We're making thread. Let's publish some info. */
        Echo(0==_EQ_Allowance
        ?   ["ATTENTION: Production allowance is 0"]
        :   ["Production allowance = ", _EQ_Allowance]
        );
        Echo([SupSurf]);

        /* Higbee stuff:
        Since threads are generated from the Z=0 plane 'upwards', the physical 'start' of external threads is at the 'end', i.e, farthest away from Z=0, in other words; the HigbeeIn cut is made at the end, whereas the HigbeeIn cut is made _first_, i.e. nearest Z=0 for internal threads. To facilitate the math and logic in the routines that follow, we effectively swap the Higbee cuts for external thread here. */
        StartHigbee=External ? HigbeeOut : HigbeeIn;
        EndHigbee=  External ? HigbeeIn  : HigbeeOut;
        /* If the helix doesn't divide in a _whole_ number of interrupted patterns, we spread the 'slop' between the ends... */
        InSlop=(_Turns*360-HigbeeIn-HigbeeOut)%(Interrupted[0]+Interrupted[1]);
        /* Create a thread helix (array of points in 3D space). */
        /*
          This is the big one, the thread profile arrives 'pre-tapered' from ThreadData.scad, but here the Higbee cut and interrupted pattern factors are added (most of the work) and finally, the profile is placed on the helix. Then the 'thread-profile-helix' is turned into a polyhedron, which--in the case of multi-start threads--is copied and finally rotated into position.
        */
        ProfileArray=[
        /* For each position around the 'turns'. */
        for (Jdex=[0:Steps])
            /* Get the run-in-out scaling factors. It's a simple (turns made)/(turns to run in-out) factor, clipped to '1'. First we check if we're in a thread start- or end-taper, then we check if we're at an interruption-taper for interrupted thread. */
            let(
                Here=Jdex/Steps,
                Done=Here*_Turns*360,
                ToDo=(1-Here)*_Turns*360,
                /* We want HigbeeSkew mirrorred at the thread end, not in the same axial direction as at the thread start. */
                Skew=Here>0.5?HigbeeSkew:-HigbeeSkew,
                /* Location on the interrupted pattern, corrected so pattern starts _after_ possible HigbeeIn. */
                InLoc=(Done-StartHigbee-InSlop/2)%(Interrupted[0]+Interrupted[1]),
                /* Are we in the InSlop or EndHigbee range at the end of the thread?. */
                EndRange=ToDo<(EndHigbee+InSlop/2)
                    ?   true:false,
                /* Higbee cut scaling factor. */
                Higbee=min(
                        StartHigbee>0?Done/StartHigbee:1,
                        EndHigbee>0?ToDo/EndHigbee:1
                    ),
                /* Interrupted threads scaling factor. */
                IntHigbee=!EndRange&&ArrayProduct(Interrupted)>0
                    ?   max(
                            0,
                            (InLoc-Interrupted[0]/2-Interrupted[1]+Interrupted[2])/Interrupted[2],
                            -(InLoc-Interrupted[0]/2-Interrupted[2])/Interrupted[2]
                        )
                    :   1,
                HigbeeFactor=min(1,Higbee,IntHigbee),
                /* Apply affine transformation matrix to the section profile (i.e. the cross-section of a single thread). This is one of the single most fantastic routines since Thales of Miletus and Allan Turing's first dreams of his machine. To be able to manipulate vertices with impunity like this is just mind-boggling; you can tell I'm impressed by this stuff...
                These translations and rotations must be carefully applied in the right order and combination to prevent a later operation influencing the result of an earlier one; a bit like PEMDAS but different ;-)
                Here, we apply the Higbee cut -- if so required. You're welcome to GTS; basically, Mr. Higbee had a bright idea that prevented fire-hose couplings from getting cross-threaded (you're usually in a hurry when you're coupling fire-hoses). This is a reasonable approximation of his idea, and named after him; he deserves the reference. Actually if there was a cost-effective way to generate this shape using machining methods (now or then), I think Mr. Higbee would be in favour of the shape achieved here, but it's too late to ask him, he's been pushing up daisies for well over a century... */
                /* Profile 'base' vertices for Higbee generation. These are the second and last-but-one vertices of the profile, i.e. the ones _at the root of the profile_, a line through these two points is--so to speak--the base of the profile. We need this base-line because:
                 * We need to bring this base to the origin for scaling the profile height with the Higbee factor.
                 * The profile may be skewed for tapered threads so it must be sheared back 'into square'.
                 * We may need to move the profile laterally for the 'skewed' Higbee. */
                Second=_Profile[1],
                LastButOne=_Profile[len(_Profile)-2],
/* Attenuator   How many parameters is too many? This one made my camel groan
                (you know, straw, camel's back...) The 'Attenuator' -- see below -- makes the Higbee 2D efect milder (read: less 'peaked'). It looks better and it's stronger, I really didn't think that needed to be in the call, but it's here just in case you _really_ want to mess with it. It's used in the next call, some 25 lines down. */
                Attenuator=0.3,
                _HProfile=AffineTransform(TransformMatrix([
                    /* Translate profile radially onto the origin, corrected for taper. */
                    [
                        -((Second+LastButOne)/2).x,
                        0,
                        0,
                    "T"],
                    /* Shear tapered profile 'upright'. */
                    [
                        [1,
                            -(Second-LastButOne).x/(Second-LastButOne).y,
                        0],
                        [0,1,0],
                        [0,0,1],
                    "SH"],
                    /* Translate profile 'vertically' to effect 'Skew'. */
                    [
                        0,
                        -Second.y*Skew,
                        0,
                    "T"],
                    /* Scale profile into Higbee cut. */
                    [
                        HigbeeFactor,
                        Higbee2D
                        ?   (HigbeeFactor+Attenuator)/(1+Attenuator)
/* Attenuator           The '2D' Higbee looks better when the Y-Factor is
                        somewhat attenuated; if that's not your style, just set the 'Attenuator' to '0' (see above), but you'll get a horribly 'peaked' cross-section in your Higbee cut. */
                        :   1,
                        1,
                        "S"],
                    /* Cancel 'Skew' translation. */
                    [
                        0,
                        Second.y*Skew,
                        0,
                    "T"],
                    /* Cancel taper shear. */
                    [
                        [1,
                            (Second-LastButOne).x/(Second-LastButOne).y,
                        0],
                        [0,1,0],
                        [0,0,1],
                    "SH"],
                    /* Cancel radial translation onto the origin. */
                    [
                        ((Second+LastButOne)/2).x,
                        0,
                        0,
                    "T"]
                    ]),
                _Profile),
                /* Keep the first and last vertex pair from the original profile, they define the support surface or 'base' of the thread, which we don't want to affect. All other vertices are taken from the 'Higbeed' profile.
                This is the reason why profiles _must_ have at least 6 vertices (see TheGHOUL/Lib/Parts/ThreadData.scad); you can't address vertices between the starting and ending pair if there aren't any... */
                HProfile=[
                    _Profile[0],
                    _Profile[1],
                    for(Hdex=[2:len(_HProfile)-3])
                        _HProfile[Hdex],
                    _Profile[len(_Profile)-2],
                    _Profile[len(_Profile)-1]
                ],
                /* Place the profile in it's final location in the helix. */
                Profile=AffineTransform(TransformMatrix([
                    /* The profile is defined on Z=0, turn it 'into the vertical'. */
                    [90,0,0,"R"],
                    [
                        /* Bring the profile out to it's radius. */
                        _PitchRadius-(_Lead*Jdex/_FN*tan(_TaperAngle))
                        ,0
                        /* The profile is defined symmetrically around the X-axis, we add '_Pitch'/2 vertically to bring the bottom of the thread up to the Z=0 datum, then we add the thread axial travel per turn. */
                        ,_Pitch/2+_Lead*Jdex/_FN,
                        "T"
                    ],
                    /* Rotate the profile to it's angular location on the helix. */
                    [0,0,360*Jdex/_FN*_Handedness,"R"]
                ]),HProfile)
            )
            Profile
        ];

        for (Idex=[0:MultiStart-1])
            /* Distribute multi-start thread helixes. */
            rotate([0,0,Idex/MultiStart*360])
            /* Clearance correction rotation, not only makes meshing internal and external threads look good in the rendering, it is of importance when for example a closure (i.e. lid or bolt-head) needs to face a certain way when the threads lock up. */
            rotate([0,0,_ClearanceContactCorrection*_Handedness*(External?1:-1)
            ])
            /* Because taper moves the thread-flanks radially, and we want to maintain the nominal diameter at max. taper OD, we move the external thread up, and the internal thread down a little (this happens in the profile generator in 'ThreadData.scad') so the intersection of nominal diameter and threadflank (still with me?) stays in the same place. This brings a tiny bit of the internal threads (at the thread start) _below_ the Z=0 plane. Here, we 'difference()' that piece off. Don't worry about it. If you want to see what I mean, make some internal threads with a ridiculous amount of taper, say 30 degrees, and comment the two MARKed lines out. There's a demo file '/TheGHOUL/DemoFiles/Img_NoCropThread.scad' demonstrating this as well. */
/* MARK */  difference(){
                /* Turn vertex cloud into a polyhedron. */
                CoverMesh(ProfileArray,Endcaps=true);
/* MARK */      translate([0,0,-_Pitch])cylinder(r=R1+_Height,h=_Pitch);
            }
    }

}

/*
  There. Not to brag, but I seriously doubt you can find a better thread generator for OpenSCAD anywhere; I'll happily wait for you to prove me wrong.
*/

// EOF ===========================================================
