/*
    +++++ ++++  +   +   +   +  +++  +++++   +++++  +++    ++++  +++ +++++
      +   +   + +   +   ++  + +   +   +       +   +   +   +   +  +  +
      +   ++++   + +    + + + +   +   +       +   +   +   +   +  +  +++
      +   +   +   +     +  ++ +   +   +       +   +   +   +   +  +  +
      +   +   +   +     +   +  +++    +       +    +++    ++++  +++ +++++

    If you're even THINKING of printing 'load bearing' threads:
    >> Get the hell out of here!
    If you're sick enough to even CONSIDER printing NPT threads for an application in ANY gas piping, or piping subjected to EVEN THE SLIGHTEST PRESSURE:
    >> You need SERIOUS help; have yourself committed!
    Have you EVER witnessed 'just a garden hose' that blew apart? Do you have kids or pets?
    * Water pressure can be up to 100 PSI in some municipalities, although 50 PSI is more common. A garden hose Internal Diameter of 3/4 inch is quite normal, this makes for a 40+ pound-force trying to tear your coupling apart, or 20+ pounds in lower pressure municipalities. If it succeeds, would you like to get slapped in the face with 40+ pounds? Didn't think so. -- These numbers are valid for a 'dead-ended' hose.
    >>  Do NOT even THINK about using printed threads for ANY pressure applications. Not even in the garden. You sick fuck.

    For all others; please be aware of the limitations of printed objects. Do NOT rely on the mechanical strength or integrity of printed objects in ANY application where failure of that object presents ANY KIND OF RISK.

    There, you have been warned, I wash my hands in innocence.
*/

/*
  (A very incomplete) collection of thread profiles:

  Metric
    - This thread has a pitch lookup dictionary for diameters M2 to M12.

  MetricFine
    - This thread has a pitch lookup dictionary for diameters M2 to M12.

  UNC
    - This thread has a pitch lookup dictionary for diameters #1 to 1/2 inch.

  UNF
    - This thread has a pitch lookup dictionary for diameters #1 to 1/2 inch.

  NPT
    - This thread has a pitch lookup dictionary for diameters 1/8 to 4 inch.

  GardenHoseThreadProfile()
    - There is ONLY ONE correct nominal diameter and pitch for garden hoses,
      therefore calls are made only specifying a conversion factor if required.

  TriangularThreadProfile()
    - Generic triangular profile, crest trim with intersection, root fillet
      with supporting surface.

  SquareThreadProfile()
    - Square profile thread (0 degree thread ThreadAngle) meshes differently than
      trapezoid threads, see additional info at the profile declaration below.

  KnuckleThreadProfile()
    - Knuckle profile thread (0 degree thread angle) meshes differently than
      trapezoid threads, see additional info at the profile declaration below.

 */

/*
  MEGA-SUPER-IMPORTANT NOTE:
  Thread profile 'dimensions' (in this file) are dimensionless (mm/mm and Inch/Inch &c.).

  This approach works well when units are consistent, calling for metric threads from a metric (i.e: millimeter based) design will yield the expected results. When incorporating an NPT connection into a metric based design however, it is necessary to convert your units accordingly as '1' in NPT thread means 'one inch' whereas in a metric thread it means 'one millimeter'. Keep this in mind, and use the 'ConversionFactor' parameter.
 */

/*
  First things first: Thread Tables.

  These are 'lookup' tables where the ThreadData() routine finds the thread geometry data for each of the defined/standardised thread diameters of each thread type.

  The FIRST element is:
  [0,MiD,MaD,ThreadAngle,TaperAngle,EQ_Allowance]
  * '0', placeholder, identifies first element.
  * MiD, Minor Diameter, fraction of height, internal diameter 'coordinate'
    relative to Pitch Diameter. Hole 'size' of internal thread.
  * MaD, Major Diameter, fraction of height, external diameter 'coordinate'
    relative to Pitch Diameter. External 'diameter' (OD) of external thread.
  * ThreadAngle, included angle between the thread profile flanks.
  * TaperAngle, included angle of the taper (e.g. NPT).
  * EQ_Allowance is a machine dependent number, i.e, it depends on
        YOUR 3D PRINTER,
    it 'compensates' for inaccuracies in your printer, and is therefore an absolute dimension (in native units), which is why it is divided by the conversion factor for thread standards that are native to the Imperial system e.g. 'UNC' &c.

    EQ_Allowance is defined in /TheGHOUL/Config.scad

  The REMAINING  elements structure is:
  [Nominal Identifier, Major Diameter, Pitch, Height]
  * Nominal Identifier, often the same as the OD but also often a completely different thing, like in 'UNC #6' thread. Identifies the thread size.
  * Major Diameter, actual OD dimension of the external thread.
  * Pitch, pitch dimension of the thread.
  * Height, height dimension of the thread profile.
   ** Most tables will _not_ contain height data as Height follows from Pitch and ThreadAngle.

  It's easy to create some exotic thread, The minimum required would be to add a 'MyExoticThread' specification:

  MyExoticThread=[[0,-7/16,7/16,55,0,EQ_Allowance]];

  Then add the following two lines in the selector function 'ThreadData()' (see below the tables):

  :   "MyExoticThread"==ThreadType
       ?    TriangularThread(NominalDiameter,Pitch,RootClearance,Clearance,ConversionFactor,MyExoticThread)

  Adding something like a buttress thread would take a bit more work, but it's all well commented, so give'r.
*/
/*
  IMPORTANT: See EQ_Allowance comments above.
*/
/* Metric threads. */
MetricThread=[["MetricThread",-2/8,3/8,60,0,EQ_Allowance],[2,2,0.4],[3,3,0.5],[4,4,0.7],[5,5,0.8],[6,6,1],[8,8,1.25],[10,10,1.5],[12,12,1.75],[14,14,2],[16,16,2],[18,18,2.5],[20,20,2.5],[22,22,2.5],[24,24,3],[27,27,3],[30,30,3.5],[33,33,3.5],[36,36,4],[39,39,4],[42,42,4.5],[45,45,4.5],[48,48,5],[52,52,5],[56,56,5.5],[60,60,5.5],[64,64,6],[68,68,6],[72,72,6],[80,80,6],[90,90,6],[100,100,6]];
MetricFineThread=[["MetricFineThread",-2/8,3/8,60,0,EQ_Allowance],[2,2,0.25],[3,3,0.35],[4,4,0.5],[5,5,0.5],[6,6,0.75],[8,8,1],[10,10,1.25],[12,12,1.5],[14,14,1.5],[16,16,1.5],[18,18,2],[20,20,2],[22,22,2],[24,24,2],[27,27,2],[30,30,3],[33,33,3],[36,36,3],[39,39,3],[42,42,4],[45,45,4],[48,48,4],[52,52,4],[56,56,4],[60,60,4],[64,64,4],[68,68,4],[72,72,4],[80,80,4],[90,90,4],[100,100,4]];
/* Some 'American' threads. */
UNCThread=[["UNCThread",-2/8,3/8,60,0,EQ_Allowance/25.4],["#1",0.073,1/64],["#2",0.086,1/56],["#3",0.099,1/48],["#4",0.112,1/40],["#5",0.125,1/40],["#6",0.138,1/32],["#8",0.164,1/32],["#10",0.19,1/24],["#12",0.216,1/24],[1/4,1/4,1/20],[5/16,5/16,1/18],[3/8,3/8,1/16],[7/16,7/16,1/14],[1/2,1/2,1/13],[9/16,9/16,1/12],[5/8,5/8,1/11],[3/4,3/4,1/10],[7/8,7/8,1/9],[1,1,1/8],[1+1/8,1+1/8,1/7],[1+1/4,1+1/4,1/7],[1+3/8,1+3/8,1/6],[1+1/2,1+1/2,1/6]];
UNFThread=[["UNFThread",-2/8,3/8,60,0,EQ_Allowance/25.4],["#1",0.073,1/72],["#2",0.086,1/64],["#3",0.099,1/56],["#4",0.112,1/48],["#5",0.125,1/44],["#6",0.138,1/40],["#8",0.164,1/36],["#10",0.19,1/32],["#12",0.216,1/28],[1/4,1/4,1/28],[5/16,5/16,1/24],[3/8,3/8,1/24],[7/16,7/16,1/20],[1/2,1/2,1/20],[9/16,9/16,1/18],[5/8,5/8,1/18],[3/4,3/4,1/16],[7/8,7/8,1/14],[1,1,1/14],[1+1/8,1+1/8,1/12],[1+1/4,1+1/4,1/12],[1+3/8,1+3/8,1/12],[1+1/2,1+1/2,1/12]];
/* NPT tapers 3/4" per foot (included angle), hence atan(3/8/12)*2. */
NPTThread=[["NPTThread",-3/8,3/8,60,atan(3/8/12)*2,EQ_Allowance/25.4],[1/8,0.405,1/27,0.068],[1/4,0.540,1/18,0.088],[3/8,0.675,1/18,0.091],[1/2,0.84,1/14,0.109],[3/4,1.05,1/14,0.113],[1,1.315,1/11.5,0.133],[1+1/4,1.66,1/11.5,0.14],[1+1/2,1.9,1/11.5,0.145],[2,2.375,1/11.5,0.154],[2+1/2,2.875,1/8,0.203],[3,3.5,1/8,0.216],[3+1/2,4,1/8,0.226],[4,4.5,1/8,0.237]];
/* Gardenhose thread is an Imperial thread, to use it in a metric design you must specify 'ConversionFactor=25.4'. */
GardenHoseThread=[["GardenHoseThread",-3/8,3/8,60,0,EQ_Allowance/25.4],["Std",1+1/16,2/23]];
/* Non-standard triangular thread. */
TriangularThread=[["TriangularThread",-2/8,3/8,60,0,EQ_Allowance]];
/* Things are getting a little exotic now... */
SquareThread=[["SquareThread",-0.5,0.5,0,0,EQ_Allowance]];
KnuckleThread=[["KnuckleThread",-0.5,0.5,30,0,EQ_Allowance]];
/* National Pipe Size ('National' here refers to the country calling itself 'the USA'.) [Nominal,OD,T]*/
NPS=[[1/8,0.405,0.068],[1/4,0.540,0.088],[3/8,0.675,0.091],[1/2,0.84,0.109],[3/4,1.05,0.113],[1,1.315,0.133],[1+1/4,1.66,0.14],[1+1/2,1.9,0.145],[2,2.375,0.154],[2+1/2,2.875,0.203],[3,3.5,0.216],[3+1/2,4,0.226],[4,4.5,0.237]];

/*
  Typical thread profile:

  Symmetric straight thread profiles are fully described by three diameters, a pitch, and an angle. Square profiles have issues, they're the black sheep with knuckle profiles as a contender for the title, we'll treat them later. Most thread types are defined by a 'dictionary' or look-up list, linking standardized outside diameters or identifiers (like '#4' thread) to pitch and crest and root dimensions of the threads.

  Thread profiles here are described by a polygon. The first and last vertex have equal and opposite Y-values of +/- Pitch/2 and X-values of +/- Height/2, depending on the gender of the thread, that is to say they are centered on the origin, with the pitch-diameter on the Y-axis, and 'symmetrically' around the X-axis, even though the profile itself may not be symmetrical. Starting at the positive extreme Y-value for external thread and at the negative extreme value for internal thread (this is to assure clockwise faces later in the mesh generator) all successive vertices should have a Y-value that is either decreasing (external thread) or increasing (internal thread) from the previous one. Essentially the profile should be convex along it's entire path. Although other profiles are thinkable*, they are not likely to be very usable.

  * There's a saying about such things: The engineers were so busy thinking about whether they _could_, they forgot to think about whether they _should_.

  The root radius offset (profile_vector[3]) is the distance from the Pitch diameter to the root fillet or supporting cylindrical surface of the thread.


     |<------------------- Y-axis
    >|-----------------|<- base radius offset
     |
     | >|--------------|<- root radius offset
     |  |              |
   --X--|--------------|---------------------------------------------
        o              |                                           ^
        |  o           |                                           |
        |     o        |<--- nominal radius                        |
        |        o     |                                           |
        |           o  |    'X' are the vertices of the profile    |
        |              X    specified clockwise from top-left      |
        |              o                                           |
   -----|--------------o----  X-axis                         Pitch |
        |              o                                           |
        |              X    profile does not _have_ to be          |
        |           o       symetrical around the X-axis           |
        |        o          but the first and last vertex          |
        |     o        MUST be [0, Pitch/2] and [0, -Pitch/2]      |
        |  o                                                       |
        o                                                          v
   --X--|------------------------------------------------------------
        o
           o

  profile vectors are built like this:
  [
      [
          [-Height/2, Pitch/2],       // #0 points of the thread section polygon
          ...,
          [-Height/2, -Pitch/2]
      ],
      Pitch,                  // #1 the thread Pitch (axial movement per turn)
      Pitch radius,           // #2 the radius at 1/2 Height of the profile
                                    where the flanks of the external and internal threads are equidistant
      root offset,            // #3 root offset
      thread ThreadAngle,     // #4 included Angle of thread flanks
      taper angle             // #5 included angle of thread taper
      allowance,              // #6 allowance (production tolerance)
      undef,                  // #7 future use
      undef,                  // #8 future use
      external                // #9 Boolean, true for external thread
  ]
 */

/*
  We need a small Fudge-Factor (FF) here to keep CGAL happy during rendering. I haven't dug into this (I'm sorry but I can't be asked) but this happens:

  Rendering Polygon Mesh using CGAL...
  ERROR: CGAL error in CGAL_Nef_polyhedron3(): CGAL ERROR: assertion violation! Expr: e->incident_sface() != SFace_const_handle() File: /usr/include/CGAL/Nef_S2/SM_const_decorator.h Line: 330

  The successive revolutions of the helix have coinciding vertices when the thread profile is exactly *Pitch* high, and it seems that CGAL doesn't like this much and falls apart. Or something. Like I said, I didn't investigate and I have no aspirations to become a CGAL guru right now. The fudge factor we use here is tiny, and provides a minuscule overlap in the successive 'wraps' of the thread-helix.
*/
FF=FrogsHair; // So fine, You can't see it...

/* ThreadData() is not a user-function. It is used by the Thread() routine, to generate the thread-profile (cross-section) for the thread-helix. */
function ThreadData(ThreadType,NominalDiameter,Pitch,ThreadAngle,Height,TaperAngle,Clearance,RootClearance,ConversionFactor)=
    let(
        // Flag presence of 'non standard' parameters to alert user that these are not required for a standardised thread-profile and will be ignored.
        NonStd=(Pitch!=undef||ThreadAngle!=undef||Height!=undef||TaperAngle!=undef)
    )
    /* Is *ThreadType* standardised?. */
    IsInArray(ThreadType[0][0], // Outside [] needed because search() treats strings--let's be nice--'peculiarly', so to say...
        [
            "UNCThread","UNFThread","MetricThread","MetricFineThread","NPTThread"
        ]
    )
    /* Call the StandardThread() profile generator and alert user in case standard parameters have been erroneously specified--they are taken from the Thread Tables--there's a reason we have _standards_. StandardThread() has a *Foo* 'black hole' to dump the Alert() into.*/
    ?   StandardThread(NominalDiameter,Clearance,ConversionFactor,ThreadType,Alert(["'",ThreadType[0][0],"' is a standardised thread, values are read from a lookup table, supplied values other than Nominal Diameter are ignored. Use 'TriangularThread' for non-standard threads."],0,NonStd))
    /* Other threads, they may have a Thread Table above, but they're not necessarily a _standardised_ thread... */
    :   "TriangularThread"==ThreadType[0][0]
        ?   TriangularThread(NominalDiameter,Pitch,ThreadAngle,TaperAngle,Clearance,ConversionFactor,ThreadType)
    :   "SquareThread"==ThreadType[0][0]
        ?   SquareThread(NominalDiameter,Pitch,Height,ThreadAngle,TaperAngle,RootClearance,Clearance,ConversionFactor,ThreadType)
    :   "KnuckleThread"==ThreadType[0][0]
        ?   KnuckleThread(NominalDiameter,Pitch,Height,ThreadAngle,TaperAngle,RootClearance,Clearance,ConversionFactor,ThreadType)
    :   "GardenHoseThread"==ThreadType[0][0]
        ?   StandardThread("Std",Clearance,ConversionFactor,ThreadType)
    /*

        Okay, so you want to be like that...

        To quickly use an exotic/non-standard triangular thread, declare it in your top level (as in the next line in this comment), and use it's name as your 'ThreadType' parameter in 'Thread()'.
        MyThreadName=[["XOTCThread",-MiD,MaD,ThreadAngle,TaperAngle,EQ_Allowance]];

        * The "XOTCThread" string in element [0][0] is the compulsory name, so it can be recognised below.
        * MiD [-0.5:0] and MaD [0:0.5] are dimensionless fractions of the thread height, see the thread tables above for more info.
        * ThreadAngle and TaperAngle are included angles, and can be '0' if they are supplied in the Thread() call.
        * EQ_Allowance can be referred to by name as it is declared in TheGHOUL/Lib/Definitions/Constants.scad, or replaced by a scalar.

        Also see the Thread Tables above.
    */
    :   "XOTCThread"==ThreadType[0][0]
        ?   TriangularThread(NominalDiameter,Pitch,ThreadAngle,TaperAngle,Clearance,ConversionFactor,ThreadType)
    /*
        If we made it here, the ThreadType called for has not-yet been defined, let's alert the user...
    */
    :   Stop(["The thread type definition has not been found in ThreadData.scad."])
;

// == STANDARD THREAD PROFILE ==================================================
/*
  StandardThread()

  Metric, Metric fine, UNC and UNF NPT and many other thread specifications share the same 60 degree angle and minor and major diameter proportions. They all use this, 'Standardthread()' routine.

  NominalIdentifier:    scalar or string, outside diameter identifier of the
                        external thread.
  ConversionFactor:     scalar.
  ThreadTable:          array, see above.

  Returns: array containing polygon of internal and external thread sections and relevant diameters. The polygon y-axis is on the pitch diameter of the thread.
*/
/*
  $Verbose=false is included in the SVG() calls here; nobody cares about foci of the root radii.
*/
function StandardThread(NominalIdentifier,Clearance,ConversionFactor,ThreadTable,Foo)=
    let(
        /* Major diameter, pitch and thread angle come from the standard, unless we're making some kind of 'custom' thread, in which case 'TriangularThread' should be used, so: No forgiveness... */
        _MajorDiameter=GetValue(NominalIdentifier,ThreadTable,1),
        _Pitch=GetValue(NominalIdentifier,ThreadTable,2),
        /* Minor Diameter distance to Pitch Diameter, a.k.a. minor offset */
        _MiD=ThreadTable[0][1],
        /* Major Diameter distance to Pitch Diameter, a.k.a. major offset */
        _MaD=ThreadTable[0][2],
        /* ThreadAngle of thread profile. */
        _ThreadAngle=ThreadTable[0][3],
        /* TaperAngle of thread. */
        _TaperAngle=ThreadTable[0][4],
        /* Allowance. */
        _Allowance=ThreadTable[0][5],
        /* Thread height follows from pitch and thread angle. See https://en.wikipedia.org/wiki/Screw_thread. */
        _Height=_Pitch/2/tan(_ThreadAngle/2),
        _PitchDiameter=_MajorDiameter-_Height*_MaD*2,

        /* Clearance and _Allowance radial and axial movements. */
        _Clr=(_Allowance+Clearance),
        _ClrR=_Clr*sin(_ThreadAngle/2),
        _ClrA=_Clr*cos(_ThreadAngle/2),

        /* Movements due to taper, radial and axial. The top half of the profile shears radially inward, the bottom half outward. The centerline of the profile is stationary. */
        _TaperRm=_Pitch/2*tan(_TaperAngle/2), // Radial movement.
        _TaperAm=_TaperRm*tan(_ThreadAngle/2), // Axial movement.

        /* Root Diameter distance to Pitch Diameter, based on the major/minor offset values, unless specified in the call.
        The calculation constructs a radius from the apex of the profile through the corners of the thread peak. The distance between the circle segment thus created and the peak, on the profile center line is used for the root clearance. */
        _MiRr=(1/2+_MiD)/cos(_ThreadAngle/2)*tan(_ThreadAngle/2), // Ext root radius -- MinorRootRadius
        _MiRc=(1/2+_MiD)/Pow2(cos(_ThreadAngle/2))-(1/2), // Ext root radius center -- MinorRootCenter
        _MaRr=(1/2-_MaD)/cos(_ThreadAngle/2)*tan(_ThreadAngle/2), // Int root radius -- MajorRootRadius
        _MaRc=(1/2)-(1/2-_MaD)/Pow2(cos(_ThreadAngle/2)), // Int root radius center -- MajorRootCenter
        _MiC=(1/2+_MiD)*tan(_ThreadAngle/2), // Half Minor crest
        _MaC=(1/2-_MaD)*tan(_ThreadAngle/2)  // Half Major crest

    )
    [
        /* Profiles are mapped with the Pitch Diameter on the Y-axis and Y-values between +/-Pitch/2 and X-values between +/-_Height/2. ConversionFactor here allows us to mix metric and imperial thread standards. See it's comment in 'Thread.scad' */
        SVG([
            "M",
            -_Height/2-_TaperRm,_Pitch/2+FF, // full height (minor) - taper radial shift at half pitch
            _Height*(_MiRc-_MiRr)-_TaperRm-_Clr,_Pitch/2+FF, // minor root - taper radial shift at half pitch
            "A",
            _Height*_MiRr+_Clr,_Height*_MiRr+_Clr, // minor root radius
            "0 0 1", // SVG arc flags
            _Height*_MiD-_TaperRm-_ClrR,_Pitch/2*(1/2-_MiD)-_ClrA, // minor crest - taper radial shift
            "L",
            _Height*_MaD-_ClrR, _Pitch/2*(1/2-_MaD)-_TaperAm-_ClrA, // major crest - taper axial shift
            _Height*_MaD-_ClrR,-_Pitch/2*(1/2-_MaD)-_TaperAm+_ClrA, // major crest - taper axial shift
            _Height*_MiD+_TaperRm-_ClrR,-_Pitch/2*(1/2-_MiD)+_ClrA, // minor crest + taper radial shift
            "A",
            _Height*_MiRr+_Clr,_Height*_MiRr+_Clr, // minor root radius
            "0 0 1", // SVG arc flags
            _Height*(_MiRc-_MiRr)+_TaperRm-_Clr,
            -_Pitch/2-FF, // minor root + taper radial shift at half pitch
            "L",
            -_Height/2+_TaperRm,-_Pitch/2-FF // full height (minor) + taper radial shift at half pitch
        ],$Verbose=false)[0]*ConversionFactor,             /* # 0 External profile */
        ArrayAdd(SVG([
            "M",
            _Height/2+_TaperRm,-_Pitch/2-FF,
            _Height*(_MaRc+_MaRr)+_TaperRm+_Clr,-_Pitch/2-FF,
            "A",
            _Height*_MaRr+_Clr,_Height*_MaRr+_Clr,
            "0 0 1",
            _Height*_MaD+_TaperRm+_ClrR,-_Pitch/2*(1/2+_MaD)+_ClrA,
            "L",
            _Height*_MiD+_ClrR,-_Pitch/2*(1/2+_MiD)+_TaperAm+_ClrA,
            _Height*_MiD+_ClrR, _Pitch/2*(1/2+_MiD)+_TaperAm-_ClrA,
            _Height*_MaD-_TaperRm+_ClrR, _Pitch/2*(1/2+_MaD)-_ClrA,
            "A",
            _Height*_MaRr+_Clr,_Height*_MaRr+_Clr,
            "0 0 1",
            _Height*(_MaRc+_MaRr)-_TaperRm+_Clr,_Pitch/2+FF,
            "L",
            _Height/2-_TaperRm,_Pitch/2+FF

        ],$Verbose=false)[0],[0,-_TaperAm])*ConversionFactor,             /* # 1 Internal profile */
        _MajorDiameter*ConversionFactor,    /* # 2 Major diameter */
        _Pitch*ConversionFactor,            /* # 3 Pitch */
        _Height*ConversionFactor,           /* # 4 Height */
        _PitchDiameter*ConversionFactor,    /* # 5 Pitch diameter */
        _MaRr,                              /* # 6 Major root radius (int) */
        _MiRr,                              /* # 7 Minor root radius (ext) */
        undef,                              /* # 8  */
        undef,                              /* # 9  */
        _ThreadAngle,                       /* #10 Thread angle */
        _TaperAngle,                        /* #11 Taper angle */
        ThreadTable[0][5]*ConversionFactor  /* #12 EQ_Allowance */
    ]
;

// == GENERAL TRIANGULAR THREAD PROFILE ========================================
/*
  TriangularThread()

  NominalIdentifier: scalar, outside diameter of the external thread.
  Pitch: scalar, provided to allow direct calls for non-standard threads.
  Height: scalar, provided to allow direct calls for non-standard threads.
  ThreadAngle: scalar, provided to allow direct calls for non-standard threads.
  RootClearance, scalar, root clearance dimension, absolute.
  ConversionFactor, scalar,
  ThreadTable, array

  Returns: array containing polygon of thread section and relevant diameters. The polygon y-axis is on the pitch diameter of the thread.
*/
/*
  $Verbose=false is included in the SVG() calls here; nobody cares about foci of the root radii.
*/
function TriangularThread(NominalIdentifier,Pitch,ThreadAngle,TaperAngle,Clearance,ConversionFactor,ThreadTable)=
    let(
        __MajorDiameter=GetValue(NominalIdentifier,ThreadTable,1),
        _MajorDiameter=undef==__MajorDiameter
        ?   NominalIdentifier
        :   __MajorDiameter,
        _Pitch=undef==Pitch
        ?   GetValue(NominalIdentifier,ThreadTable,2)
        :   Pitch,
        _MiD=ThreadTable[0][1], // dimensionless
        _MaD=ThreadTable[0][2], // dimensionless
        _ThreadAngle=undef==ThreadAngle
        ?   ThreadTable[0][3]
        :   ThreadAngle,
        _TaperAngle=undef==TaperAngle
        ?   ThreadTable[0][4]
        :   TaperAngle,
        _Allowance=ThreadTable[0][5],
        _Height=_Pitch/2/tan(_ThreadAngle/2),
        _PitchDiameter=_MajorDiameter-_Height*_MaD*2,
        _Clr=(_Allowance+Clearance),
        _ClrR=_Clr*sin(_ThreadAngle/2),
        _ClrA=_Clr*cos(_ThreadAngle/2),
        _TaperRm=_Pitch/2*tan(_TaperAngle/2), // Radial movement.
        _TaperAm=_TaperRm*tan(_ThreadAngle/2), // Axial movement.
        _MiRr=(1/2+_MiD)/cos(_ThreadAngle/2)*tan(_ThreadAngle/2), // Ext root radius - dimensionless
        _MiRc=(1/2+_MiD)/Sqrt(cos(_ThreadAngle/2))-(1/2), // Ext root radius center - dimensionless
        _MaRr=(1/2-_MaD)/cos(_ThreadAngle/2)*tan(_ThreadAngle/2), // Int root radius - dimensionless
        _MaRc=(1/2)-(1/2-_MaD)/Sqrt(cos(_ThreadAngle/2)), // Int root radius center - dimensionless
        _MaC=(1/2-_MaD)*tan(_ThreadAngle/2), // Half Major crest width - dimensionless
        _MiC=(1/2+_MiD)*tan(_ThreadAngle/2)  // Half Minor crest width - dimensionless

    )
    [
        SVG([
            "M",
            -_Height/2-_TaperRm,_Pitch/2+FF, // full height (minor) - taper radial shift at half pitch
            _Height*(_MiRc-_MiRr)-_TaperRm-_Clr,_Pitch/2+FF, // minor root - taper radial shift at half pitch
            "A",
            _Height*_MiRr+_Clr,_Height*_MiRr+_Clr, // minor root radius
            "0 0 1", // SVG arc flags
            _Height*_MiD-_TaperRm-_ClrR,_Pitch/2*(1/2-_MiD)-_ClrA, // minor crest - taper radial shift
            "L",
            _Height*_MaD-_ClrR, _Pitch/2*(1/2-_MaD)-_TaperAm-_ClrA, // major crest - taper axial shift
            _Height*_MaD-_ClrR,-_Pitch/2*(1/2-_MaD)-_TaperAm+_ClrA, // major crest - taper axial shift
            _Height*_MiD+_TaperRm-_ClrR,-_Pitch/2*(1/2-_MiD)+_ClrA, // minor crest + taper radial shift
            "A",
            _Height*_MiRr+_Clr,_Height*_MiRr+_Clr, // minor root radius
            "0 0 1", // SVG arc flags
            _Height*(_MiRc-_MiRr)+_TaperRm-_Clr,-_Pitch/2-FF, // minor root + taper radial shift at half pitch
            "L",
            -_Height/2+_TaperRm,-_Pitch/2-FF // full height (minor) + taper radial shift at half pitch
        ],$Verbose=false)[0]*ConversionFactor,                /* # 0 External profile */
        ArrayAdd(SVG([
            "M",
            _Height/2+_TaperRm,-_Pitch/2-FF,
            _Height*(_MaRc+_MaRr)+_TaperRm+_Clr,-_Pitch/2-FF,
            "A",
            _Height*_MaRr+_Clr,_Height*_MaRr+_Clr,
            "0 0 1",
            _Height*_MaD+_TaperRm+_ClrR,-_Pitch/2*(1/2+_MaD)+_ClrA,
            "L",
            _Height*_MiD+_ClrR,-_Pitch/2*(1/2+_MiD)+_TaperAm+_ClrA,
            _Height*_MiD+_ClrR, _Pitch/2*(1/2+_MiD)+_TaperAm-_ClrA,
            _Height*_MaD-_TaperRm+_ClrR, _Pitch/2*(1/2+_MaD)-_ClrA,
            "A",
            _Height*_MaRr+_Clr,_Height*_MaRr+_Clr,
            "0 0 1",
            _Height*(_MaRc+_MaRr)-_TaperRm+_Clr,_Pitch/2+FF,
            "L",
            _Height/2-_TaperRm,_Pitch/2+FF
        ],$Verbose=false)[0],[0,-_TaperAm])*ConversionFactor, /* # 1 Internal profile */
        _MajorDiameter*ConversionFactor,       /* # 2 Major diameter */
        _Pitch*ConversionFactor,               /* # 3 Pitch */
        _Height*ConversionFactor,              /* # 4 Height */
        _PitchDiameter*ConversionFactor,       /* # 5 Pitch diameter */
        _MaRr,                                 /* # 6 Major root radius (int) */
        _MiRr,                                 /* # 7 Minor root radius (ext) */
        undef,                                 /* # 8  */
        undef,                                 /* # 9  */
        _ThreadAngle,                          /* #10 Thread angle */
        _TaperAngle,                           /* #11 Taper angle */
        ThreadTable[0][5]*ConversionFactor     /* #12 EQ_Allowance */
    ]
;

// == SQUARE THREAD PROFILE ====================================================
/*
  SquareThreadProfile()

  NominalDiameter: scalar, outside diameter of the external thread.
  Pitch: scalar, pitch.
  Height: scalar, profile height.
  Internal: boolean, 'true' for internal threads.
*/

/* TODO - add some radii to the profile, e.g. Pitch/10. */

function SquareThread(NominalDiameter,Pitch,Height,ThreadAngle,TaperAngle,RootClearance,Clearance,ConversionFactor,ThreadTable)=
    let(
        __MajorDiameter=GetValue(NominalDiameter,ThreadTable,1),
        _MajorDiameter=undef==__MajorDiameter
        ?   NominalDiameter
        :   __MajorDiameter,
        _Pitch=undef==Pitch
        ?   GetValue(NominalDiameter,ThreadTable,2)
        :   Pitch,
        _Height=undef==Height
        ?   GetValue(NominalDiameter,ThreadTable,3)
        :   Height,
        _MiD=ThreadTable[0][1],
        _MaD=ThreadTable[0][2],
        _ThreadAngle=0,/* _ThreadAngle=undef==ThreadAngle // Leave this here, may use it later (ACME?).
        ?   ThreadTable[0][3]
        :   ThreadAngle, */
        _TaperAngle=undef==TaperAngle
        ?   ThreadTable[0][4]
        :   TaperAngle,
        _Allowance=ThreadTable[0][5],
        _PitchDiameter=_MajorDiameter-_Height*_MaD*2,
        _Clr=(_Allowance+Clearance),
        /* Root Diameter distance to Pitch Diameter, negative RootClearance is less material is more clearance. */
        _MiR=_MiD-(RootClearance+_Clr)/Height,
        _MaR=_MaD+(RootClearance+_Clr)/Height
    )
    [
        /* Profiles are mapped with the Pitch Diameter on the Y-axis and Y-values between +/-Pitch/2 and X-values between +/-_Height/2. */
        /* ConversionFactor here allows us to mix metric and imperial thread standards. See it's comment in 'Thread.scad' */
        [
            [_Height*(_MiR), _Pitch/4-_Clr],
            /* Because of the Higbee routine in 'Threads()', a profile _must_ consist of at least 5 vertices, the first and last are repeated here to get a total of 6. */
            [_Height*(_MiR), _Pitch/4-_Clr],
            [_Height*_MaD,   _Pitch/4-_Clr],
            [_Height*_MaD,  -_Pitch/4+_Clr],
            [_Height*(_MiR),-_Pitch/4+_Clr],
            [_Height*(_MiR),-_Pitch/4+_Clr]
        ]*ConversionFactor,                 /* # 0 External profile */
        [
            [_Height*(_MaR),-_Pitch/4+_Clr],
            /* See comment in external profile. */
            [_Height*(_MaR),-_Pitch/4+_Clr],
            [_Height*_MiD,  -_Pitch/4+_Clr],
            [_Height*_MiD,   _Pitch/4-_Clr],
            [_Height*(_MaR), _Pitch/4-_Clr],
            [_Height*(_MaR), _Pitch/4-_Clr]
        ]*ConversionFactor,                 /* # 1 Internal profile */
        _MajorDiameter*ConversionFactor,    /* # 2 Major diameter */
        _Pitch*ConversionFactor,            /* # 3 Pitch */
        _Height*ConversionFactor,           /* # 4 Height */
        _PitchDiameter*ConversionFactor,    /* # 5 Pitch diameter */
        0,                                  /* # 6 Major root radius (int) */
        0,                                  /* # 7 Minor root radius (ext) */
        undef,                              /* # 8  */
        undef,                              /* # 9  */
        _ThreadAngle,                       /* #10 Thread angle */
        _TaperAngle,                        /* #11 Taper angle */
        ThreadTable[0][5]*ConversionFactor  /* #12 EQ_Allowance */
    ]
;

// == KNUCKLE THREAD PROFILE ===================================================
/*
  KnuckleThreadProfile()

  NominalDiameter: scalar, outside diameter of the external thread.
  Pitch: scalar, pitch.
  Internal: boolean, 'true' for internal threads.
*/

/* TODO: profile generation, svg arcs etc. A lot. */
/* TODO: profile generation, svg arcs etc. A lot. */
/* TODO: profile generation, svg arcs etc. A lot. */
/* TODO: profile generation, svg arcs etc. A lot. */
/* TODO: profile generation, svg arcs etc. A lot. */
/* TODO: profile generation, svg arcs etc. A lot. */

function KnuckleThread(NominalDiameter,Pitch,Height,ThreadAngle,TaperAngle,RootClearance,Clearance,ConversionFactor,ThreadTable)=
    let(
        __MajorDiameter=GetValue(NominalDiameter,ThreadTable,1),
        _MajorDiameter=undef==__MajorDiameter
        ?   NominalDiameter
        :   __MajorDiameter,
        _Pitch=undef==Pitch
        ?   GetValue(NominalDiameter,ThreadTable,2)
        :   Pitch,
        _Height=undef==Height
        ?   GetValue(NominalDiameter,ThreadTable,3)
        :   Height,
        _MiD=ThreadTable[0][1],
        _MaD=ThreadTable[0][2],
        _ThreadAngle=0,/* _ThreadAngle=undef==ThreadAngle // Leave this here, may use it later (ACME?).
        ?   ThreadTable[0][3]
        :   ThreadAngle, */
        _TaperAngle=undef==TaperAngle
        ?   ThreadTable[0][4]
        :   TaperAngle,
        _Allowance=ThreadTable[0][5],
        _PitchDiameter=_MajorDiameter-_Height*_MaD*2,
        _Clr=(_Allowance+Clearance),
        /* Root Diameter distance to Pitch Diameter, negative RootClearance is less material is more clearance. */
        _MiR=_MiD-(RootClearance+_Clr)/Height,
        _MaR=_MaD+(RootClearance+_Clr)/Height
    )
    [
        /* Profiles are mapped with the Pitch Diameter on the Y-axis and Y-values between +/-Pitch/2 and X-values between +/-_Height/2. */
        /* ConversionFactor here allows us to mix metric and imperial thread standards. See it's comment in 'Thread.scad' */
        [
            [_Height*(_MiR), _Pitch/4-_Clr],
            /* Because of the Higbee routine in 'Threads()', a profile _must_ consist of at least 5 vertices, the first and last are repeated here to get a total of 6. */
            [_Height*(_MiR), _Pitch/4-_Clr],
            [_Height*_MaD,   _Pitch/4-_Clr],
            [_Height*_MaD,  -_Pitch/4+_Clr],
            [_Height*(_MiR),-_Pitch/4+_Clr],
            [_Height*(_MiR),-_Pitch/4+_Clr]
        ]*ConversionFactor,                 /* # 0 External profile */
        [
            [_Height*(_MaR),-_Pitch/4+_Clr],
            /* See comment in external profile. */
            [_Height*(_MaR),-_Pitch/4+_Clr],
            [_Height*_MiD,  -_Pitch/4+_Clr],
            [_Height*_MiD,   _Pitch/4-_Clr],
            [_Height*(_MaR), _Pitch/4-_Clr],
            [_Height*(_MaR), _Pitch/4-_Clr]
        ]*ConversionFactor,                 /* # 1 Internal profile */
        _MajorDiameter*ConversionFactor,    /* # 2 Major diameter */
        _Pitch*ConversionFactor,            /* # 3 Pitch */
        _Height*ConversionFactor,           /* # 4 Height */
        _PitchDiameter*ConversionFactor,    /* # 5 Pitch diameter */
        0,                                  /* # 6 Major root radius (int) */
        0,                                  /* # 7 Minor root radius (ext) */
        undef,                              /* # 8  */
        undef,                              /* # 9  */
        _ThreadAngle,                       /* #10 Thread angle */
        _TaperAngle,                        /* #11 Taper angle */
        ThreadTable[0][5]*ConversionFactor  /* #12 EQ_Allowance */
    ]
;

