/*
    Simply call any printable character (ascii 33-126) by it's Font Name and string like this:

    Font definitions like OpenSans_Semibold.scad are stored in TheGHOUL/Lib/SVG/Fonts.

    SVG_Glyph(OpenSans_Semibold,"A");

    or any other character with it's UniCode Hex-number like this:

    SVG_Glyph(OpenSans_Semibold,"00A9");

*/

/* The function returns a GHOUL 'shape' definition [points,paths], which can be used with polygon():  */
function SVG_Glyph(Font,Character,Size=[1,1])=
    let(
        // Generate UniCode for 'Printable characters'.
        UniCode=len(Character)==1
        // Character is a single string character.
        ?   UniCodeMapString(Character)[0]
        // 'Character is a 'UniCode' (hopefully).
        :   Character,
        // Find the named record in the collection.
        Record=search([UniCode],Font)[0],
        // Trap non-existing name.
        Glyph=Record==[]
        ?   Stop(["Font ",Font[0][0]," does not contain the glyph for UniCode ",UniCode,"."])//TODO return .notdef character?
        :   Font[Record],
        PTuple=SVG(Glyph[1],Glyph[2],Size),
        XAdv=Glyph[5]
    )
    // Return the glyph as polygon points, paths, and glyph-width, in a tuple [points,paths,x-advance].
    [
        PTuple[0],PTuple[1],
        XAdv
    ]
;

/* For the actual polygon call the module: */
module SVG_Glyph(Font,Character,Size=[1,1]){

    // get the [points,paths] tuple
    Polygon=SVG_Glyph(Font,Character,Size);
    // generate the polygon
    polygon(Polygon[0],Polygon[1]);

}

// HELPER FUNCTIONS AND MODULES ================================================

/*
  Returns the kern-value for a glyph pair of 'Font'.
  The kern-value is the number of units U2 has to shift left (closer to U1).
  Looks for U2 in the kern-list of U1 and returns the associated kern-value.
  Returns '0' if there is no entry.
*/
function KernValue(Font,U1,U2)=
    let(
        /* get the 'U1' record from 'Font'. */
        Record=search([U1],Font)[0],
        /* get the kern-list from 'Record'. */
        KList=Font[Record][4],
        /* Look for 'U2' in each entry in the kern-list, if found, extract the kern-value for that entry. */
        KValue=[
                for(I=[0:Len(KList)])
                    let(Foo=search([U2],KList[I])[0])
                    if(Foo!=[])KList[I][0]
            ][0]
//        ,Foo=Print([U1," - ",Record])
    )
    /* Return kern value from the entry or '0' if there is none. */
    KValue==undef
    ?   0
    :   KValue
;

/* Parse a string or a tuple consisting of strings of 'Printable characters' and UniCode glyph Hex identifiers into a list of just UniCode glyph code points:

echo(GlyphList(["215B"," is 12.5 %"]));

ECHO: ["215B", "0020", "0069", "0073", "0020", "0031", "0032", "002E", "0035", "0020", "0025"]

echo(GlyphList("The GHOUL"));

ECHO: ["0054", "0068", "0065", "0020", "0047", "0048", "004F", "0055", "004C"]

*/
function GlyphList(TextTuple)=
    _GlyphList(TextTuple,Len(TextTuple),[])
;
/* The recursive 'helper', i.e., the real workhorse... */
function _GlyphList(TextTuple,Idex,Result)=
    let(
        Element=TextTuple[Idex]
    )
    Idex<0
    ?   Result
    :   IsUniCode(Element)
        ?   _GlyphList(TextTuple,Idex-1,concat(
                [Element],
                Result)
            )
        :   _GlyphList(TextTuple,Idex-1,concat(
                UniCodeMapString(Element),
                Result)
            )
;

function GlyphData(Font,GlyphList,Index=0,Location=0,Points=0,Result=[])=
    GlyphList[Index]==undef // we're past Len()
    ?   [
            [
                for(Glyph=Result)
                    for(Points=Glyph[0])
                        Points
            ],
            [
                for(Glyph=Result)
                    for(Paths=Glyph[1])
                        Paths
            ]
        ]
    :   let(
            Data=SVG_Glyph(Font,GlyphList[Index]),
            Kern=Index>0
            ?   KernValue(Font,GlyphList[Index-1],GlyphList[Index])
            :   0,
            _Location=Location+Data[2]-Kern,
            _Points=Points+len(Data[0])
        )
        GlyphData(Font,GlyphList,Index+1,_Location,_Points,
            concat(Result,
                [[
                    ArrayAdd(Data[0],[Location,0]),
                    [
                        for(Path=Data[1])
                            ArrayAdd(Path,Points)
                    ]
                ]]
            )
        )
;

/*
 * TODO
 *
  Text along any curve, letters are placed descend-base-center-mean-ascent on curve. ('center' is between base and mean, or 'half-x' height.)

  Curve must be continuous and preferably differentiable at all points (radii, no sharp kinks).

  Letters can be 'fixed' or 'flex', i.e. they keep their shape and proportions or distort along the shape of the curve (60s psychedelic text).

  Place text between two curves, curves are base and ascend, text now also fluctuates in height with curve distance as height-parameter.
*/

function SVG_Text(Font,TextTuple)=
    GlyphData(Font,GlyphList(TextTuple))
;
module SVG_Text(Font,TextTuple){
    Poly=GlyphData(Font,GlyphList(TextTuple));
    polygon(Poly[0],Poly[1]);
}


//echo(SVG_Glyph(OpenSans_Semibold,"T"));
//echo(SVG_Text(OpenSans_Semibold,"TT"));
