/*
  Generate a regular star polygon shape.

  Regular star polygons have at least 5 points.

  Schlaefli:    Real [0:1] ratio between internal and external point radii,
                not a regular star polygon. Probably.
                Integer [1:...] Schläfli turning number.
                A Schäfli turning number of 1 yields a regular polygon,
                e.g {5/1} is a pentagon.
                A turning number of 2 yields the first possible regular star
                polygon, e.g. {5/2} is a regular star pentagon or pentagram.
                See: https://en.wikipedia.org/wiki/Star_polygon
*/
function Star(Points,Radius,Schlaefli=2,Centered=true)=
    let(
        IRadius=Schlaefli<1
        ?   Schlaefli*Radius
        :   IsInteger(Schlaefli)&&Schlaefli<Points/2
            ?   Radius*(cos(180/Points)-sin(180/Points)*
                sin(180*(Schlaefli-1)/Points)/cos(180*(Schlaefli-1)/Points))
            :   Stop(str("Schlaefli turning number must be integer less than Points/2; ",Schlaefli," is not valid.")),
        Step=360/Points,
        OffSet=StarOffset(Points,Radius),
        Foo=Echo(["Star Offset is ",OffSet]),
        Center=
            Centered
            ?   [0,0]
            : OffSet
    )
    [
        FlattenArray([
            for(I=[0:Points-1])
            [Radius*[cos(I*Step),sin(I*Step)]+Center,
            IRadius*[cos((I+0.5)*Step),sin((I+0.5)*Step)]+Center]
        ]),
        [
            [for (I=[0:2*Points-1]) I]
        ]

    ]
;
module Star(Points,Radius,Schlaefli=2,Centered=true){
    // Prevent multiple echos from the function.
    Foo=Star(Points,Radius,Schlaefli,Centered);
    polygon(Foo[0],Foo[1],2);
}
/*
  Helper function to quickly access Star dimensions
*/
function StarOffset(Points,Radius)=
    IsEven(Points)
    ?   [Radius,Radius*cos(180/Points)]
    :   [Radius*cos(180/Points),Radius*cos(90/Points)]
;



