/*
  Accepts three vertices or two vectors.
  Returns a unit vector with the direction of the line that bisects the _acute angle_ between the two vectors or the two line segments starting at V2 and ending in V1 and V3.

  If the three vertices or the two vectors are colinear, the result will be the LPerpVector of segment V2 to V3 or vector V2.
*/
function BisectAcute(V1,V2,V3)=
    let(
        /* Turn three vertices into two vectors. */
        V1C=V3==undef?V1:V1-V2,
        V2C=V3==undef?V2:V3-V2,
        /* Rotate V1 onto the positive X-Axis. */
        V2R=AffineRotate([0,0,-atan2(V1C.y,V1C.x)],V2C),
        /* Find angle for V2, this is the test-basis for the different cases. */
        V2RA=atan2(V2R.y,V2R.x),
        /* Find solution for non-limit cases. */
        UU=UnitVector(V1C)+UnitVector(V2C)
    )
    /* Limit case, opposite vectors, return normal vector. */
    V2RA==180
    ?   LPerpVector(V2C)
        /* Limit case, identical vectors, return same direction vector. */
    :   V2RA==0
        ?   UnitVector(V1C)
            /* Regular case. */
        :   UnitVector(UU)
;
/* Because BisectAcute() is the default method, here's the default alias. */
function Bisect(V1,V2,V3)=
    BisectAcute(V1,V2,V3)
;

/* BisectObtuse() is, not surprisingly, the reverse of... */
function BisectObtuse(V1,V2,V3)=
    -BisectAcute(V1,V2,V3)
;

/* Always bisects to the left of the path V1-V2-V3. If the vertices are successive vertices, in CCW order of a polygon boundary, the bisector will always point _into_ the polygon. */
function BisectLeft(V1,V2,V3)=
    let(
        /* Turn three vertices into two vectors. */
        V1C=V3==undef?V1:V1-V2,
        V2C=V3==undef?V2:V3-V2,
        /* Rotate V1 onto the positive X-Axis. */
        V2R=AffineRotate([0,0,-atan2(V1C.y,V1C.x)],V2C),
        /* Find angle for V2, this is the test-basis for the different cases. */
        V2RA=atan2(V2R.y,V2R.x),
        /* Find solution for non-limit cases. */
        UU=UnitVector(V1C)+UnitVector(V2C)
    )
    /* Limit case, opposite vectors, return normal vector. */
    V2RA==180
    ?   LPerpVector(V2C)
        /* Limit case, identical vectors, return same direction vector. */
    :   V2RA==0
        ?   UnitVector(V1C)
            /* Acute case. */
        :   V2RA>0
            ?   -UnitVector(UU)
            /* Obtuse case. */
            :   UnitVector(UU)
;

/* BisectRight  is, not surprisingly, the reverse of.... */
function BisectRight(V1,V2,V3)=
    -BisectLeft(V1,V2,V3)
;

//==============================================================================


