/*
  Display a function's Domain and Range in a graph, with graph-paper background.
  This requires the function to be declared in the top level name-space
  i.e: in the main body of the main file, not inside a module &c.

  The function MUST be named 'Fx' and the declaration must be
  similar to this:

  | function Fx(X)=sin(90*X)+2;

  The call can be as simple as:

  | ShowFx([0,2]);

  Domain    : Declared like a pair of scalars [0,10].
  DotSize   : Size of the points in the graph.
  GPFactor  : Graph Paper line Factor, bigger is thinner.

  Note: The 'GraphPaper()' module compensates the line thickness for
  View-Point distance ($vpd) but it needs a refreshed preview or render
  after every zoom operation to do so.
*/
/* TODO: *Origin* parameter, if true, origin is included in graph paper even when domain and range don't get there. */
module ShowFx(Domain=[0,1],Resolution=100,DotSize=0.05,GPLFactor=100,Connect=true,Margin=undef){

/* Make a list of vertices. */
Array=[
        for(Idex=[0:Resolution])
            let(X=Domain.x+(Domain.y-Domain.x)*Idex/Resolution)
            [X,Fx(X)]
    ];
/* Get the range of the graph [Ymin:Ymax]. */
Range=ArrayDomain(Array);
/* Size. */
SX=Domain.y-Domain.x;
SY=Range.y.y-Range.x.y;
/* Check order of magnitude of largest size. */
Order=OrderOfMagnitude(max(SX,SY));
/* Divisions and final size. */
Major=pow(10,Order);
Minor=Major/10;
/* Set the margin (graph-paper size beyond the graph). */
MA=undef==Margin
    ?   Minor
    :   Margin;

/* Generate some graph paper... */
GraphPaper(Major=Major,Minor=Minor,
    Size=[[Domain.x-MA,Range.x.y-MA],[Domain.y+MA,Range.y.y+MA]],
    LineFactor=GPLFactor,Color=GPG,Alpha=0.3);

/* And generate the actual curve. */
translate([0,0,DotSize])
ShowCurve(Curve=Array,Radius=DotSize,Frac=1,Connect=Connect,Closed=false);
}
